package main

import (
	"context"
	"log"
	"net"
	"net/http"
	"net/http/httputil"
	"net/url"
	"os"
	"os/signal"
	"path"
	"path/filepath"
	"strings"
	"syscall"
	"time"

	blossomsvc "ipfstr/internal/blossom"
	"ipfstr/internal/config"
	httpapi "ipfstr/internal/http"
	"ipfstr/internal/ipfs"
	nostrworker "ipfstr/internal/nostr"
	"ipfstr/internal/store"
)

func main() {
	cfg, err := config.LoadFromEnv()
	if err != nil {
		log.Fatalf("config error: %v", err)
	}

	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	storeConn, err := store.Open(cfg.DatabasePath)
	if err != nil {
		log.Fatalf("database error: %v", err)
	}
	defer storeConn.Close()

	ipfsClient := ipfs.New(cfg.IPFSAPIURL, cfg.IPFSHTTPTimeout, cfg.IPFSPinTimeout)
	if !ipfsClient.IsUp() {
		log.Printf("warning: IPFS API not reachable at %s", cfg.IPFSAPIURL)
	}

	handlers := httpapi.New(cfg, storeConn, ipfsClient)
	publicDir := resolvePublicDir(cfg.PublicDir)

	blossomSvc, err := blossomsvc.New(cfg, storeConn, ipfsClient)
	if err != nil {
		log.Fatalf("blossom error: %v", err)
	}

	nostrWorker, err := nostrworker.New(cfg, storeConn, ipfsClient, log.Default())
	if err != nil && cfg.NostrPubkey != "" {
		log.Fatalf("nostr worker error: %v", err)
	}
	if err == nil && nostrWorker.Enabled() {
		nostrWorker.Start(ctx)
		log.Printf("nostr worker enabled")
	} else if cfg.NostrPubkey == "" {
		log.Printf("nostr worker disabled (NPUB not set)")
	}

	publicMux := http.NewServeMux()
	publicMux.HandleFunc("/upload", handlers.Upload)
	publicMux.HandleFunc("/logo.png", servePublicFile(publicDir, "logo.png"))
	publicMux.HandleFunc("/favicon-32.png", servePublicFile(publicDir, "favicon-32.png"))
	publicMux.HandleFunc("/favicon-16.png", servePublicFile(publicDir, "favicon-16.png"))
	publicMux.Handle("/ipfs/", newIPFSProxy(cfg.IPFSGatewayProxyURL))
	publicMux.Handle("/", publicHandler(blossomSvc.Handler))

	publicAddr := ":" + cfg.Port
	if cfg.BindAddr != "" {
		publicAddr = net.JoinHostPort(cfg.BindAddr, cfg.Port)
	}

	publicServer := &http.Server{
		Addr:              publicAddr,
		Handler:           withCORS(cfg.CorsOrigins, withLogging(publicMux)),
		ReadHeaderTimeout: 10 * time.Second,
	}

	log.Printf("ipfstr public listening on http://%s", publicAddr)
	go func() {
		if err := publicServer.ListenAndServe(); err != nil && err != http.ErrServerClosed {
			log.Fatalf("public server error: %v", err)
		}
	}()

	adminServer := (*http.Server)(nil)
	adminAddr := ""
	if cfg.AdminPort != "" && cfg.AdminPort != "0" {
		adminMux := http.NewServeMux()
		adminMux.HandleFunc("/health", handlers.Health)
		adminMux.HandleFunc("/status", handlers.Status)
		adminMux.HandleFunc("/upload", handlers.Upload)
		adminMux.HandleFunc("/nostr", handlers.Nostr)
		adminMux.HandleFunc("/api/ipfs", handlers.IPFSInfo)
		adminMux.HandleFunc("/api/ipfs/gc", handlers.IPFSGC)
		adminMux.HandleFunc("/api/ipfs/pins", handlers.IPFSPins)
		adminMux.HandleFunc("/api/ipfs/pin/add", handlers.IPFSPinAdd)
		adminMux.HandleFunc("/api/ipfs/pin/rm", handlers.IPFSPinRemove)
		adminMux.HandleFunc("/api/ipfs/swarm/peers", handlers.IPFSSwarmPeers)
		adminMux.HandleFunc("/api/ipfs/swarm/connect", handlers.IPFSSwarmConnect)
		adminMux.HandleFunc("/api/ipfs/swarm/disconnect", handlers.IPFSSwarmDisconnect)
		adminMux.HandleFunc("/api/pins", handlers.Pins)
		adminMux.HandleFunc("/api/pins/ignore", handlers.PinIgnore)
		adminMux.HandleFunc("/api/pins/unignore", handlers.PinUnignore)
		adminMux.HandleFunc("/api/uploads", handlers.Uploads)

		adminMux.Handle("/", adminHandler(publicDir))
		if publicDir == "" {
			log.Printf("warning: public directory not found; admin UI disabled")
		}

		adminAddr = ":" + cfg.AdminPort
		if cfg.AdminBindAddr != "" {
			adminAddr = net.JoinHostPort(cfg.AdminBindAddr, cfg.AdminPort)
		}

		adminServer = &http.Server{
			Addr:              adminAddr,
			Handler:           withLogging(adminMux),
			ReadHeaderTimeout: 10 * time.Second,
		}

		log.Printf("ipfstr admin listening on http://%s", adminAddr)
		go func() {
			if err := adminServer.ListenAndServe(); err != nil && err != http.ErrServerClosed {
				log.Fatalf("admin server error: %v", err)
			}
		}()
	} else {
		log.Printf("ipfstr admin disabled (ADMIN_PORT=0)")
	}

	waitForShutdown(cancel)
	shutdownCtx, shutdownCancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer shutdownCancel()
	_ = publicServer.Shutdown(shutdownCtx)
	if adminServer != nil {
		_ = adminServer.Shutdown(shutdownCtx)
	}
}

func resolvePublicDir(path string) string {
	if path == "" {
		return ""
	}
	if info, err := os.Stat(path); err == nil && info.IsDir() {
		abs, err := filepath.Abs(path)
		if err == nil {
			return abs
		}
		return path
	}

	wd, err := os.Getwd()
	if err != nil {
		return ""
	}

	candidate := filepath.Join(wd, path)
	if info, err := os.Stat(candidate); err == nil && info.IsDir() {
		abs, err := filepath.Abs(candidate)
		if err == nil {
			return abs
		}
		return candidate
	}

	return ""
}

func rootHandler(publicDir string, blossomHandler http.Handler) http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		if tryServeStatic(publicDir, w, r) {
			return
		}
		blossomHandler.ServeHTTP(w, r)
	})
}

func adminHandler(publicDir string) http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		if tryServeStatic(publicDir, w, r) {
			return
		}
		http.NotFound(w, r)
	})
}

func publicHandler(blossomHandler http.Handler) http.Handler {
	landing := `<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1.0" />
  <title>ipfstr</title>
  <link rel="icon" type="image/png" sizes="32x32" href="/favicon-32.png" />
  <link rel="icon" type="image/png" sizes="16x16" href="/favicon-16.png" />
  <style>
    body { font-family: "IBM Plex Sans", Arial, sans-serif; background: #f5f7fb; color: #0e141b; margin: 0; }
    main { max-width: 720px; margin: 10vh auto; padding: 32px; background: #fff; border-radius: 16px; box-shadow: 0 12px 30px rgba(0,0,0,0.08); }
    h1 { margin: 0 0 12px 0; font-size: 30px; }
    p { margin: 8px 0; color: #56616f; }
    code { background: #eef1f6; padding: 2px 6px; border-radius: 6px; }
    .brand { display: flex; align-items: center; gap: 16px; margin-bottom: 12px; }
    .brand img { width: 68px; height: 68px; border-radius: 16px; object-fit: cover; box-shadow: 0 10px 20px rgba(14,20,27,0.12); border: 1px solid rgba(14,20,27,0.08); }
    .muted { color: #56616f; font-size: 14px; }
  </style>
</head>
<body>
  <main>
    <div class="brand">
      <img src="/logo.png" alt="ipfstr logo" />
      <h1>ipfstr Blossom endpoint</h1>
    </div>
    <p>This is the public Blossom/media endpoint for Nostr clients.</p>
    <p>Use <code>/upload</code> for direct uploads and the Blossom routes for clients.</p>
    <p class="muted">Instance: <span id="instance">unknown</span></p>
  </main>
  <script>
    const instance = document.getElementById("instance");
    if (instance) {
      instance.textContent = window.location.origin;
    }
  </script>
</body>
</html>`

	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		if strings.Contains(strings.ToLower(r.Header.Get("Connection")), "upgrade") {
			blossomHandler.ServeHTTP(w, r)
			return
		}
		if r.Method == http.MethodGet && r.URL.Path == "/" {
			w.Header().Set("Content-Type", "text/html; charset=utf-8")
			w.WriteHeader(http.StatusOK)
			_, _ = w.Write([]byte(landing))
			return
		}
		blossomHandler.ServeHTTP(w, r)
	})
}

func tryServeStatic(publicDir string, w http.ResponseWriter, r *http.Request) bool {
	if publicDir == "" {
		return false
	}

	cleanPath := path.Clean("/" + r.URL.Path)
	cleanPath = strings.TrimPrefix(cleanPath, "/")
	if strings.HasSuffix(r.URL.Path, "/") || cleanPath == "" {
		cleanPath = path.Join(cleanPath, "index.html")
	}

	target := filepath.Join(publicDir, filepath.FromSlash(cleanPath))
	absTarget, err := filepath.Abs(target)
	if err != nil {
		return false
	}
	absPublic, err := filepath.Abs(publicDir)
	if err != nil {
		return false
	}
	if absTarget != absPublic && !strings.HasPrefix(absTarget, absPublic+string(os.PathSeparator)) {
		return false
	}

	info, err := os.Stat(absTarget)
	if err != nil || info.IsDir() {
		return false
	}

	http.ServeFile(w, r, absTarget)
	return true
}

func servePublicFile(publicDir string, filename string) http.HandlerFunc {
	return func(w http.ResponseWriter, r *http.Request) {
		if publicDir == "" {
			http.NotFound(w, r)
			return
		}
		target := filepath.Join(publicDir, filename)
		info, err := os.Stat(target)
		if err != nil || info.IsDir() {
			http.NotFound(w, r)
			return
		}
		http.ServeFile(w, r, target)
	}
}

func withLogging(next http.Handler) http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		start := time.Now()
		next.ServeHTTP(w, r)
		duration := time.Since(start)
		log.Printf("%s %s %s", r.Method, r.URL.Path, duration)
	})
}

func withCORS(origins []string, next http.Handler) http.Handler {
	allowAll := len(origins) == 0 || (len(origins) == 1 && origins[0] == "*")
	originSet := make(map[string]struct{})
	for _, origin := range origins {
		originSet[origin] = struct{}{}
	}

	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		origin := r.Header.Get("Origin")
		if allowAll {
			w.Header().Set("Access-Control-Allow-Origin", "*")
		} else if origin != "" {
			if _, ok := originSet[origin]; ok {
				w.Header().Set("Access-Control-Allow-Origin", origin)
				w.Header().Set("Vary", "Origin")
			}
		}

		w.Header().Set("Access-Control-Allow-Methods", "GET, POST, PUT, OPTIONS")
		requestedHeaders := r.Header.Get("Access-Control-Request-Headers")
		if requestedHeaders != "" {
			w.Header().Set("Access-Control-Allow-Headers", requestedHeaders)
			w.Header().Set("Vary", "Origin, Access-Control-Request-Headers")
		} else {
			w.Header().Set("Access-Control-Allow-Headers", "Content-Type, Authorization, X-SHA-256, X-Content-Length")
		}
		w.Header().Set("Access-Control-Max-Age", "3600")

		if r.Method == http.MethodOptions {
			w.WriteHeader(http.StatusNoContent)
			return
		}

		next.ServeHTTP(w, r)
	})
}

func newIPFSProxy(target string) http.Handler {
	parsed, err := url.Parse(target)
	if err != nil || parsed.Scheme == "" || parsed.Host == "" {
		log.Printf("warning: invalid IPFS_GATEWAY_PROXY_URL: %s", target)
		return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			http.Error(w, "IPFS gateway proxy not configured", http.StatusServiceUnavailable)
		})
	}

	proxy := httputil.NewSingleHostReverseProxy(parsed)
	originalDirector := proxy.Director
	proxy.Director = func(r *http.Request) {
		originalDirector(r)
		r.Host = parsed.Host
	}
	return proxy
}

func waitForShutdown(cancel context.CancelFunc) {
	signalChan := make(chan os.Signal, 1)
	signal.Notify(signalChan, syscall.SIGINT, syscall.SIGTERM)
	<-signalChan
	cancel()
}
