package blossomsvc

import (
	"bytes"
	"context"
	"encoding/json"
	"fmt"
	"io"
	"net/http"
	"net/url"
	"os"
	"runtime"
	"strings"
	"sync"

	"ipfstr/internal/config"
	"ipfstr/internal/ipfs"
	"ipfstr/internal/store"

	"github.com/fiatjaf/eventstore/sqlite3"
	"github.com/fiatjaf/khatru"
	"github.com/fiatjaf/khatru/blossom"
)

// Service wraps the blossom relay and response mapper.
type Service struct {
	Handler http.Handler
	Relay   *khatru.Relay
}

// New initializes Blossom/Khatru with IPFS storage.
func New(cfg config.Config, storeConn *store.Store, ipfsClient *ipfs.Client) (*Service, error) {
	if storeConn == nil {
		return nil, fmt.Errorf("store not initialized")
	}
	if ipfsClient == nil {
		return nil, fmt.Errorf("ipfs client not initialized")
	}

	backend := &sqlite3.SQLite3Backend{DatabaseURL: cfg.DatabasePath}
	if err := backend.Init(); err != nil {
		return nil, fmt.Errorf("init eventstore: %w", err)
	}

	relay := khatru.NewRelay()
	relay.StoreEvent = append(relay.StoreEvent, backend.SaveEvent)
	relay.QueryEvents = append(relay.QueryEvents, backend.QueryEvents)
	relay.CountEvents = append(relay.CountEvents, backend.CountEvents)
	relay.DeleteEvent = append(relay.DeleteEvent, backend.DeleteEvent)
	relay.ReplaceEvent = append(relay.ReplaceEvent, backend.ReplaceEvent)

	serviceURL := cfg.ServiceURL
	if strings.TrimSpace(serviceURL) == "" {
		serviceURL = fmt.Sprintf("http://localhost:%s", cfg.Port)
	}

	bl := blossom.New(relay, serviceURL)
	bl.Store = blossom.EventStoreBlobIndexWrapper{Store: backend, ServiceURL: bl.ServiceURL}

	bl.StoreBlob = append(bl.StoreBlob, func(ctx context.Context, sha256 string, ext string, body []byte) error {
		addResult, err := ipfsClient.Add(ctx, bytes.NewReader(body), cfg.PinUploads)
		if err != nil {
			return err
		}
		if err := storeConn.UpsertMapping(ctx, sha256, addResult.Hash, ext); err != nil {
			return err
		}
		return nil
	})

	bl.LoadBlob = append(bl.LoadBlob, func(ctx context.Context, sha256 string, ext string) (io.ReadSeeker, error) {
		cid, _, err := storeConn.GetMapping(ctx, sha256)
		if err != nil {
			return nil, fmt.Errorf("mapping lookup: %w", err)
		}
		reader, err := ipfsClient.Cat(ctx, cid)
		if err != nil {
			return nil, err
		}
		defer reader.Close()

		tmp, err := os.CreateTemp("", "ipfstr-blob-*")
		if err != nil {
			return nil, fmt.Errorf("create temp file: %w", err)
		}
		if _, err := io.Copy(tmp, reader); err != nil {
			_ = tmp.Close()
			_ = os.Remove(tmp.Name())
			return nil, fmt.Errorf("copy ipfs data: %w", err)
		}
		if _, err := tmp.Seek(0, io.SeekStart); err != nil {
			_ = tmp.Close()
			_ = os.Remove(tmp.Name())
			return nil, fmt.Errorf("seek temp file: %w", err)
		}
		return newTempReadSeeker(tmp), nil
	})

	mapped := modifyBlossomResponse(relay, storeConn, cfg.GatewayURL, cfg.GatewayFallbackURL)
	return &Service{Handler: mapped, Relay: relay}, nil
}

func modifyBlossomResponse(relay http.Handler, storeConn *store.Store, gatewayURL string, fallbackURL string) http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		if isWebSocketRequest(r) {
			relay.ServeHTTP(w, r)
			return
		}

		redir := &responseRedirected{ResponseWriter: w}
		if r.Method == http.MethodGet {
			maybeRedirectToGateway(redir, r, storeConn, gatewayURL)
			if redir.done {
				return
			}
		}

		captured := &responseCapturer{
			ResponseWriter: w,
			statusCode:     http.StatusOK,
			body:           &bytes.Buffer{},
			headers:        make(http.Header),
		}

		relay.ServeHTTP(captured, r)

		bodyBytes := captured.body.Bytes()
		if captured.statusCode < 200 || captured.statusCode >= 300 || len(bodyBytes) == 0 {
			writeCaptured(w, captured)
			return
		}

		modified, ok := tryModifyJSONResponse(r.Context(), storeConn, gatewayURL, fallbackURL, bodyBytes)
		if !ok {
			writeCaptured(w, captured)
			return
		}

		copyHeaders(w, captured.headers)
		w.WriteHeader(captured.statusCode)
		_, _ = w.Write(modified)
	})
}

func isWebSocketRequest(r *http.Request) bool {
	connection := strings.ToLower(r.Header.Get("Connection"))
	upgrade := strings.ToLower(r.Header.Get("Upgrade"))
	return strings.Contains(connection, "upgrade") && upgrade == "websocket"
}

func maybeRedirectToGateway(w http.ResponseWriter, r *http.Request, storeConn *store.Store, gatewayURL string) {
	path := strings.TrimPrefix(r.URL.Path, "/")
	if !strings.Contains(path, ".") {
		return
	}
	lastDot := strings.LastIndex(path, ".")
	if lastDot <= 0 {
		return
	}
	sha256 := path[:lastDot]
	ext := path[lastDot:]
	if len(sha256) != 64 {
		return
	}

	cid, _, err := storeConn.GetMapping(r.Context(), sha256)
	if err != nil || cid == "" {
		return
	}

	filename := "file" + ext
	gatewayURLWithFile := gatewayURL + cid + "?filename=" + url.QueryEscape(filename)
	http.Redirect(w, r, gatewayURLWithFile, http.StatusFound)
}

func tryModifyJSONResponse(ctx context.Context, storeConn *store.Store, gatewayURL string, fallbackURL string, body []byte) ([]byte, bool) {
	bodyStr := strings.TrimSpace(string(body))
	if bodyStr == "" {
		return nil, false
	}

	if strings.HasPrefix(bodyStr, "[") {
		var arr []map[string]interface{}
		if err := json.Unmarshal(body, &arr); err != nil {
			return nil, false
		}
		modified := false
		for i := range arr {
			if enrichItem(ctx, storeConn, gatewayURL, fallbackURL, arr[i]) {
				modified = true
			}
		}
		if modified {
			out, err := json.Marshal(arr)
			if err == nil {
				return out, true
			}
		}
		return nil, false
	}

	if strings.Contains(bodyStr, "\n") && strings.Contains(bodyStr, "\"sha256\"") {
		lines := strings.Split(strings.ReplaceAll(bodyStr, "\r\n", "\n"), "\n")
		var outLines []string
		modified := false
		for _, line := range lines {
			line = strings.TrimSpace(line)
			if line == "" {
				continue
			}
			var item map[string]interface{}
			if err := json.Unmarshal([]byte(line), &item); err != nil {
				continue
			}
			if enrichItem(ctx, storeConn, gatewayURL, fallbackURL, item) {
				modified = true
			}
			encoded, err := json.Marshal(item)
			if err == nil {
				outLines = append(outLines, string(encoded))
			}
		}
		if modified && len(outLines) > 0 {
			return []byte(strings.Join(outLines, "\n") + "\n"), true
		}
		return nil, false
	}

	if strings.HasPrefix(bodyStr, "{") {
		var obj map[string]interface{}
		if err := json.Unmarshal(body, &obj); err != nil {
			return nil, false
		}
		if enrichItem(ctx, storeConn, gatewayURL, fallbackURL, obj) {
			out, err := json.Marshal(obj)
			if err == nil {
				return out, true
			}
		}
	}

	return nil, false
}

func enrichItem(ctx context.Context, storeConn *store.Store, gatewayURL string, fallbackURL string, item map[string]interface{}) bool {
	sha, ok := item["sha256"].(string)
	if !ok || sha == "" {
		return false
	}

	cid, ext, err := storeConn.GetMapping(ctx, sha)
	if err != nil || cid == "" {
		return false
	}

	item["cid"] = cid
	filename := "file"
	if ext != "" {
		filename += ext
	}
	item["url"] = buildGatewayURL(gatewayURL, cid, filename)
	if fallbackURL != "" {
		item["fallback_url"] = buildGatewayURL(fallbackURL, cid, filename)
	}
	return true
}

func buildGatewayURL(baseURL string, cid string, filename string) string {
	if baseURL == "" || cid == "" {
		return ""
	}
	urlValue := baseURL + cid
	if strings.TrimSpace(filename) == "" {
		return urlValue
	}
	return urlValue + "?filename=" + url.QueryEscape(filename)
}

func copyHeaders(w http.ResponseWriter, headers http.Header) {
	for key, values := range headers {
		for _, value := range values {
			w.Header().Add(key, value)
		}
	}
}

func writeCaptured(w http.ResponseWriter, captured *responseCapturer) {
	copyHeaders(w, captured.headers)
	w.WriteHeader(captured.statusCode)
	_, _ = w.Write(captured.body.Bytes())
}

// responseCapturer captures responses for modification.
type responseCapturer struct {
	http.ResponseWriter
	statusCode int
	body       *bytes.Buffer
	headers    http.Header
}

func (rc *responseCapturer) Header() http.Header {
	return rc.headers
}

func (rc *responseCapturer) WriteHeader(code int) {
	rc.statusCode = code
}

func (rc *responseCapturer) Write(b []byte) (int, error) {
	return rc.body.Write(b)
}

// responseRedirected tracks if a redirect happened.
type responseRedirected struct {
	http.ResponseWriter
	done bool
}

func (rr *responseRedirected) WriteHeader(statusCode int) {
	if statusCode >= 300 && statusCode < 400 {
		rr.done = true
	}
	rr.ResponseWriter.WriteHeader(statusCode)
}

type tempReadSeeker struct {
	file *os.File
	path string
	once sync.Once
}

func newTempReadSeeker(file *os.File) *tempReadSeeker {
	trs := &tempReadSeeker{
		file: file,
		path: file.Name(),
	}
	runtime.SetFinalizer(trs, func(item *tempReadSeeker) {
		_ = item.Close()
	})
	return trs
}

func (trs *tempReadSeeker) Read(p []byte) (int, error) {
	return trs.file.Read(p)
}

func (trs *tempReadSeeker) Seek(offset int64, whence int) (int64, error) {
	return trs.file.Seek(offset, whence)
}

func (trs *tempReadSeeker) Close() error {
	var err error
	trs.once.Do(func() {
		err = trs.file.Close()
		_ = os.Remove(trs.path)
	})
	return err
}
