package config

import (
	"fmt"
	"os"
	"strconv"
	"strings"
	"time"
)

// Config holds runtime configuration.
type Config struct {
	Port                 string
	BindAddr             string
	AdminPort            string
	AdminBindAddr        string
	IPFSAPIURL           string
	IPFSHTTPTimeout      time.Duration
	IPFSPinTimeout       time.Duration
	IPFSGatewayProxyURL  string
	DatabasePath         string
	GatewayURL           string
	GatewayFallbackURL   string
	FileLimitBytes       int64
	StorageMax           string
	PinUploads           bool
	HealthMaxMemoryMB    int
	HealthMaxGoroutines  int
	PublicDir            string
	ServiceURL           string
	NostrPubkey          string
	NostrRelays          []string
	NostrDiscoveryEvery  time.Duration
	NostrPinnerMinDelay  time.Duration
	NostrPinnerMaxDelay  time.Duration
	NostrLookback        time.Duration
	CorsOrigins          []string
}

// LoadFromEnv reads configuration from environment variables.
func LoadFromEnv() (Config, error) {
	cfg := Config{}

	cfg.Port = getEnvOrDefault("PORT", "3334")
	cfg.BindAddr = getEnvOrDefault("BIND_ADDR", "0.0.0.0")
	cfg.AdminPort = getEnvOrDefault("ADMIN_PORT", "3335")
	cfg.AdminBindAddr = getEnvOrDefault("ADMIN_BIND_ADDR", "127.0.0.1")
	cfg.IPFSAPIURL = getEnvOrDefault("IPFS_API_URL", "http://127.0.0.1:5002")
	cfg.IPFSHTTPTimeout = getEnvDuration("IPFS_HTTP_TIMEOUT", 5*time.Minute)
	cfg.IPFSPinTimeout = getEnvDuration("IPFS_PIN_TIMEOUT", cfg.IPFSHTTPTimeout)
	cfg.IPFSGatewayProxyURL = getEnvOrDefault("IPFS_GATEWAY_PROXY_URL", "http://ipfs:8080")
	cfg.DatabasePath = getEnvOrDefault("DATABASE_PATH", "./ipfstr.db")
	cfg.GatewayURL = getEnvOrDefault("IPFS_GATEWAY_URL", "https://dweb.link/ipfs/")
	cfg.GatewayFallbackURL = getEnvOrDefault("IPFS_GATEWAY_FALLBACK_URL", "")
	cfg.StorageMax = getEnvOrDefault("STORAGE_MAX", "200GB")
	cfg.PinUploads = getEnvBool("PIN_UPLOADS", true)
	cfg.PublicDir = getEnvOrDefault("PUBLIC_DIR", "./public")
	cfg.ServiceURL = strings.TrimSpace(os.Getenv("SERVICE_URL"))

	fileLimitRaw := getEnvOrDefault("FILE_LIMIT", "5GB")
	fileLimitBytes, err := parseBytes(fileLimitRaw)
	if err != nil {
		return cfg, fmt.Errorf("invalid FILE_LIMIT: %w", err)
	}
	cfg.FileLimitBytes = fileLimitBytes

	cfg.HealthMaxMemoryMB = getEnvInt("HEALTHCHECK_MAX_MEMORY_MB", 0)
	cfg.HealthMaxGoroutines = getEnvInt("HEALTHCHECK_MAX_GOROUTINES", 0)

	if !strings.HasSuffix(cfg.GatewayURL, "/") {
		cfg.GatewayURL += "/"
	}
	if cfg.GatewayFallbackURL != "" && !strings.HasSuffix(cfg.GatewayFallbackURL, "/") {
		cfg.GatewayFallbackURL += "/"
	}

	cfg.NostrPubkey = strings.TrimSpace(getEnvOrDefault("NPUB", ""))
	if cfg.NostrPubkey == "" {
		cfg.NostrPubkey = strings.TrimSpace(getEnvOrDefault("NOSTR_PUBKEY", ""))
	}
	cfg.NostrRelays = parseCSV(getEnvOrDefault("NOSTR_RELAYS", ""))
	cfg.NostrDiscoveryEvery = getEnvDuration("NOSTR_DISCOVERY_INTERVAL", 7*time.Minute)
	cfg.NostrPinnerMinDelay = getEnvDuration("NOSTR_PINNER_MIN_DELAY", 30*time.Second)
	cfg.NostrPinnerMaxDelay = getEnvDuration("NOSTR_PINNER_MAX_DELAY", 100*time.Second)
	cfg.NostrLookback = getEnvDuration("NOSTR_LOOKBACK", 30*24*time.Hour)
	cfg.CorsOrigins = parseCSV(getEnvOrDefault("CORS_ORIGINS", "*"))

	return cfg, nil
}

func getEnvOrDefault(key, fallback string) string {
	if value := strings.TrimSpace(os.Getenv(key)); value != "" {
		return os.ExpandEnv(value)
	}
	return os.ExpandEnv(fallback)
}

func getEnvInt(key string, fallback int) int {
	value := strings.TrimSpace(os.Getenv(key))
	if value == "" {
		return fallback
	}
	parsed, err := strconv.Atoi(value)
	if err != nil {
		return fallback
	}
	return parsed
}

func getEnvBool(key string, fallback bool) bool {
	value := strings.TrimSpace(os.Getenv(key))
	if value == "" {
		return fallback
	}
	switch strings.ToLower(value) {
	case "1", "true", "yes", "y", "on":
		return true
	case "0", "false", "no", "n", "off":
		return false
	default:
		return fallback
	}
}

func getEnvDuration(key string, fallback time.Duration) time.Duration {
	value := strings.TrimSpace(os.Getenv(key))
	if value == "" {
		return fallback
	}
	parsed, err := time.ParseDuration(value)
	if err != nil {
		return fallback
	}
	return parsed
}

func parseCSV(input string) []string {
	if strings.TrimSpace(input) == "" {
		return nil
	}
	parts := strings.Split(input, ",")
	var cleaned []string
	for _, part := range parts {
		item := strings.TrimSpace(part)
		if item != "" {
			cleaned = append(cleaned, item)
		}
	}
	return cleaned
}

// parseBytes parses strings like "5GB", "50MB", "100KB".
func parseBytes(input string) (int64, error) {
	trimmed := strings.TrimSpace(strings.ToUpper(input))
	if trimmed == "" {
		return 0, fmt.Errorf("empty size")
	}

	units := map[string]int64{
		"B":  1,
		"KB": 1024,
		"MB": 1024 * 1024,
		"GB": 1024 * 1024 * 1024,
		"TB": 1024 * 1024 * 1024 * 1024,
	}

	var numberPart string
	var unitPart string
	for i, r := range trimmed {
		if (r < '0' || r > '9') && r != '.' {
			numberPart = trimmed[:i]
			unitPart = strings.TrimSpace(trimmed[i:])
			break
		}
	}

	if numberPart == "" {
		numberPart = trimmed
	}

	if unitPart == "" {
		unitPart = "B"
	}

	value, err := strconv.ParseFloat(numberPart, 64)
	if err != nil {
		return 0, fmt.Errorf("invalid number: %w", err)
	}
	multiplier, ok := units[unitPart]
	if !ok {
		return 0, fmt.Errorf("unknown unit: %s", unitPart)
	}

	return int64(value * float64(multiplier)), nil
}
