package httpapi

import (
	"crypto/sha256"
	"encoding/hex"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"mime"
	"net/http"
	"net/url"
	"os"
	"path/filepath"
	"runtime"
	"strconv"
	"strings"
	"time"

	"ipfstr/internal/config"
	"ipfstr/internal/ipfs"
	"ipfstr/internal/store"
)

// Handlers groups HTTP endpoints.
type Handlers struct {
	cfg   config.Config
	store *store.Store
	ipfs  *ipfs.Client
}

// New creates handlers.
func New(cfg config.Config, store *store.Store, ipfsClient *ipfs.Client) *Handlers {
	return &Handlers{cfg: cfg, store: store, ipfs: ipfsClient}
}

// Health reports service health.
func (h *Handlers) Health(w http.ResponseWriter, r *http.Request) {
	checks := map[string]map[string]interface{}{}
	status := "healthy"
	statusCode := http.StatusOK

	if h.store == nil {
		status = "unhealthy"
		statusCode = http.StatusServiceUnavailable
		checks["database"] = map[string]interface{}{ "status": "unhealthy", "error": "store not initialized" }
	} else {
		checks["database"] = map[string]interface{}{ "status": "healthy" }
	}

	if h.ipfs != nil && h.ipfs.IsUp() {
		checks["ipfs"] = map[string]interface{}{ "status": "healthy" }
	} else {
		status = "unhealthy"
		statusCode = http.StatusServiceUnavailable
		checks["ipfs"] = map[string]interface{}{ "status": "unhealthy", "error": "IPFS API not accessible" }
	}

	var mem runtime.MemStats
	runtime.ReadMemStats(&mem)
	allocMB := int(mem.Alloc / 1024 / 1024)
	memStatus := "healthy"
	if h.cfg.HealthMaxMemoryMB > 0 && allocMB > h.cfg.HealthMaxMemoryMB {
		status = "unhealthy"
		statusCode = http.StatusServiceUnavailable
		memStatus = "unhealthy"
	}
	checks["memory"] = map[string]interface{}{
		"status":         memStatus,
		"alloc_mb":       allocMB,
		"total_alloc_mb": mem.TotalAlloc / 1024 / 1024,
		"sys_mb":         mem.Sys / 1024 / 1024,
		"num_gc":         mem.NumGC,
		"max_mb":         h.cfg.HealthMaxMemoryMB,
	}

	goroutines := runtime.NumGoroutine()
	gorStatus := "healthy"
	if h.cfg.HealthMaxGoroutines > 0 && goroutines > h.cfg.HealthMaxGoroutines {
		status = "unhealthy"
		statusCode = http.StatusServiceUnavailable
		gorStatus = "unhealthy"
	}
	checks["goroutines"] = map[string]interface{}{
		"status": gorStatus,
		"count":  goroutines,
		"max":    h.cfg.HealthMaxGoroutines,
	}

	payload := map[string]interface{}{
		"status":    status,
		"checks":    checks,
		"timestamp": time.Now().Unix(),
	}

	writeJSON(w, statusCode, payload)
}

// Status returns basic configuration info.
func (h *Handlers) Status(w http.ResponseWriter, r *http.Request) {
	payload := map[string]interface{}{
		"status":        "ok",
		"timestamp":     time.Now().UTC().Format(time.RFC3339),
		"gateway_url":   h.cfg.GatewayURL,
		"gateway_fallback_url": h.cfg.GatewayFallbackURL,
		"file_limit":    h.cfg.FileLimitBytes,
		"storage_max":   h.cfg.StorageMax,
		"ipfs_available": h.ipfs != nil && h.ipfs.IsUp(),
	}

	writeJSON(w, http.StatusOK, payload)
}

// Nostr reports discovery/pinner status.
func (h *Handlers) Nostr(w http.ResponseWriter, r *http.Request) {
	if h.cfg.NostrPubkey == "" {
		writeJSON(w, http.StatusOK, map[string]interface{}{
			"enabled": false,
			"reason":  "NPUB not set",
		})
		return
	}

	lastRun, _ := h.store.GetState(r.Context(), "nostr_last_run")
	lastError, _ := h.store.GetState(r.Context(), "nostr_last_error")

	var lastRunTS int64
	if lastRun != "" {
		lastRunTS, _ = strconv.ParseInt(lastRun, 10, 64)
	}

	stats, err := h.store.GetPinStats(r.Context())
	if err != nil {
		writeJSON(w, http.StatusInternalServerError, map[string]string{
			"error":   "failed to load pin stats",
			"message": err.Error(),
		})
		return
	}

	writeJSON(w, http.StatusOK, map[string]interface{}{
		"enabled":          true,
		"pubkey":           h.cfg.NostrPubkey,
		"relays":           h.cfg.NostrRelays,
		"last_run":         lastRunTS,
		"last_error":       lastError,
		"pin_stats":        stats,
		"discovery_every":  h.cfg.NostrDiscoveryEvery.String(),
		"pinner_min_delay": h.cfg.NostrPinnerMinDelay.String(),
		"pinner_max_delay": h.cfg.NostrPinnerMaxDelay.String(),
	})
}

// IPFSInfo returns node stats for the admin UI.
func (h *Handlers) IPFSInfo(w http.ResponseWriter, r *http.Request) {
	if h.ipfs == nil {
		writeJSON(w, http.StatusOK, map[string]interface{}{
			"available": false,
			"error":     "ipfs client not initialized",
		})
		return
	}

	stats, err := h.ipfs.NodeInfo(r.Context())
	stats.Available = h.ipfs.IsUp()
	if err != nil {
		stats.Error = err.Error()
	}

	writeJSON(w, http.StatusOK, stats)
}

// IPFSGC triggers IPFS garbage collection.
func (h *Handlers) IPFSGC(w http.ResponseWriter, r *http.Request) {
	if r.Method != http.MethodPost {
		writeJSON(w, http.StatusMethodNotAllowed, map[string]string{"error": "method not allowed"})
		return
	}
	if h.ipfs == nil {
		writeJSON(w, http.StatusServiceUnavailable, map[string]string{
			"error": "ipfs client not initialized",
		})
		return
	}

	removed, err := h.ipfs.RepoGC(r.Context())
	if err != nil {
		writeJSON(w, http.StatusBadGateway, map[string]string{
			"error":   "ipfs gc failed",
			"message": err.Error(),
		})
		return
	}

	writeJSON(w, http.StatusOK, map[string]interface{}{
		"success": true,
		"removed": removed,
	})
}

// IPFSPins lists pinned CIDs from the IPFS node.
func (h *Handlers) IPFSPins(w http.ResponseWriter, r *http.Request) {
	if r.Method != http.MethodGet {
		writeJSON(w, http.StatusMethodNotAllowed, map[string]string{"error": "method not allowed"})
		return
	}
	if h.ipfs == nil {
		writeJSON(w, http.StatusServiceUnavailable, map[string]string{
			"error": "ipfs client not initialized",
		})
		return
	}

	pinType := strings.TrimSpace(r.URL.Query().Get("type"))
	if pinType == "" {
		pinType = "recursive"
	}
	cid := strings.TrimSpace(r.URL.Query().Get("cid"))
	if cid == "" {
		cid = strings.TrimSpace(r.URL.Query().Get("arg"))
	}

	pins, err := h.ipfs.PinLs(r.Context(), pinType, cid)
	if err != nil {
		writeJSON(w, http.StatusBadGateway, map[string]string{
			"error":   "ipfs pin ls failed",
			"message": err.Error(),
		})
		return
	}

	writeJSON(w, http.StatusOK, map[string]interface{}{
		"success": true,
		"type":    pinType,
		"count":   len(pins),
		"pins":    pins,
	})
}

// IPFSPinAdd pins a CID on the IPFS node.
func (h *Handlers) IPFSPinAdd(w http.ResponseWriter, r *http.Request) {
	if r.Method != http.MethodPost {
		writeJSON(w, http.StatusMethodNotAllowed, map[string]string{"error": "method not allowed"})
		return
	}
	if h.ipfs == nil {
		writeJSON(w, http.StatusServiceUnavailable, map[string]string{
			"error": "ipfs client not initialized",
		})
		return
	}

	cid, recursive, err := readCIDRequest(r)
	if err != nil {
		writeJSON(w, http.StatusBadRequest, map[string]string{
			"error":   "invalid cid",
			"message": err.Error(),
		})
		return
	}

	if err := h.ipfs.PinWithOptions(r.Context(), cid, recursive); err != nil {
		writeJSON(w, http.StatusBadGateway, map[string]string{
			"error":   "ipfs pin add failed",
			"message": err.Error(),
		})
		return
	}

	writeJSON(w, http.StatusOK, map[string]interface{}{
		"success":   true,
		"cid":       cid,
		"recursive": recursive,
	})
}

// IPFSPinRemove unpins a CID on the IPFS node.
func (h *Handlers) IPFSPinRemove(w http.ResponseWriter, r *http.Request) {
	if r.Method != http.MethodPost {
		writeJSON(w, http.StatusMethodNotAllowed, map[string]string{"error": "method not allowed"})
		return
	}
	if h.ipfs == nil {
		writeJSON(w, http.StatusServiceUnavailable, map[string]string{
			"error": "ipfs client not initialized",
		})
		return
	}

	cid, recursive, err := readCIDRequest(r)
	if err != nil {
		writeJSON(w, http.StatusBadRequest, map[string]string{
			"error":   "invalid cid",
			"message": err.Error(),
		})
		return
	}

	if err := h.ipfs.Unpin(r.Context(), cid, recursive); err != nil {
		writeJSON(w, http.StatusBadGateway, map[string]string{
			"error":   "ipfs pin rm failed",
			"message": err.Error(),
		})
		return
	}

	writeJSON(w, http.StatusOK, map[string]interface{}{
		"success":   true,
		"cid":       cid,
		"recursive": recursive,
	})
}

// IPFSSwarmPeers lists connected swarm peers.
func (h *Handlers) IPFSSwarmPeers(w http.ResponseWriter, r *http.Request) {
	if r.Method != http.MethodGet {
		writeJSON(w, http.StatusMethodNotAllowed, map[string]string{"error": "method not allowed"})
		return
	}
	if h.ipfs == nil {
		writeJSON(w, http.StatusServiceUnavailable, map[string]string{
			"error": "ipfs client not initialized",
		})
		return
	}

	peers, err := h.ipfs.SwarmPeers(r.Context())
	if err != nil {
		writeJSON(w, http.StatusBadGateway, map[string]string{
			"error":   "ipfs swarm peers failed",
			"message": err.Error(),
		})
		return
	}

	writeJSON(w, http.StatusOK, map[string]interface{}{
		"success": true,
		"count":   len(peers),
		"peers":   peers,
	})
}

// IPFSSwarmConnect connects to a swarm multiaddr.
func (h *Handlers) IPFSSwarmConnect(w http.ResponseWriter, r *http.Request) {
	if r.Method != http.MethodPost {
		writeJSON(w, http.StatusMethodNotAllowed, map[string]string{"error": "method not allowed"})
		return
	}
	if h.ipfs == nil {
		writeJSON(w, http.StatusServiceUnavailable, map[string]string{
			"error": "ipfs client not initialized",
		})
		return
	}

	addr, err := readAddrRequest(r)
	if err != nil {
		writeJSON(w, http.StatusBadRequest, map[string]string{
			"error":   "invalid address",
			"message": err.Error(),
		})
		return
	}

	result, err := h.ipfs.SwarmConnect(r.Context(), addr)
	if err != nil {
		writeJSON(w, http.StatusBadGateway, map[string]string{
			"error":   "ipfs swarm connect failed",
			"message": err.Error(),
		})
		return
	}

	writeJSON(w, http.StatusOK, map[string]interface{}{
		"success": true,
		"addr":    addr,
		"result":  result,
	})
}

// IPFSSwarmDisconnect disconnects from a swarm multiaddr.
func (h *Handlers) IPFSSwarmDisconnect(w http.ResponseWriter, r *http.Request) {
	if r.Method != http.MethodPost {
		writeJSON(w, http.StatusMethodNotAllowed, map[string]string{"error": "method not allowed"})
		return
	}
	if h.ipfs == nil {
		writeJSON(w, http.StatusServiceUnavailable, map[string]string{
			"error": "ipfs client not initialized",
		})
		return
	}

	addr, err := readAddrRequest(r)
	if err != nil {
		writeJSON(w, http.StatusBadRequest, map[string]string{
			"error":   "invalid address",
			"message": err.Error(),
		})
		return
	}

	result, err := h.ipfs.SwarmDisconnect(r.Context(), addr)
	if err != nil {
		writeJSON(w, http.StatusBadGateway, map[string]string{
			"error":   "ipfs swarm disconnect failed",
			"message": err.Error(),
		})
		return
	}

	writeJSON(w, http.StatusOK, map[string]interface{}{
		"success": true,
		"addr":    addr,
		"result":  result,
	})
}

// Pins returns pin history and stats.
func (h *Handlers) Pins(w http.ResponseWriter, r *http.Request) {
	if h.store == nil {
		writeJSON(w, http.StatusServiceUnavailable, map[string]string{
			"error": "store not initialized",
		})
		return
	}

	limit, offset := parsePagination(r, 50, 200)
	pins, err := h.store.ListPins(r.Context(), limit, offset)
	if err != nil {
		writeJSON(w, http.StatusInternalServerError, map[string]string{
			"error":   "failed to list pins",
			"message": err.Error(),
		})
		return
	}

	total, err := h.store.CountPins(r.Context())
	if err != nil {
		writeJSON(w, http.StatusInternalServerError, map[string]string{
			"error":   "failed to count pins",
			"message": err.Error(),
		})
		return
	}

	stats, _ := h.store.GetPinStats(r.Context())

	writeJSON(w, http.StatusOK, map[string]interface{}{
		"success": true,
		"pins":    pins,
		"stats":   stats,
		"pagination": map[string]interface{}{
			"limit":   limit,
			"offset":  offset,
			"total":   total,
			"hasMore": offset+limit < total,
		},
	})
}

// PinIgnore marks a pin as ignored.
func (h *Handlers) PinIgnore(w http.ResponseWriter, r *http.Request) {
	if r.Method != http.MethodPost {
		writeJSON(w, http.StatusMethodNotAllowed, map[string]string{"error": "method not allowed"})
		return
	}
	if h.store == nil {
		writeJSON(w, http.StatusServiceUnavailable, map[string]string{
			"error": "store not initialized",
		})
		return
	}

	cid, _, err := readCIDRequest(r)
	if err != nil {
		writeJSON(w, http.StatusBadRequest, map[string]string{
			"error":   "invalid cid",
			"message": err.Error(),
		})
		return
	}

	if err := h.store.IgnorePin(r.Context(), cid); err != nil {
		writeJSON(w, http.StatusInternalServerError, map[string]string{
			"error":   "failed to ignore pin",
			"message": err.Error(),
		})
		return
	}

	writeJSON(w, http.StatusOK, map[string]interface{}{
		"success": true,
		"cid":     cid,
		"status":  "ignored",
	})
}

// PinUnignore resets a pin to pending.
func (h *Handlers) PinUnignore(w http.ResponseWriter, r *http.Request) {
	if r.Method != http.MethodPost {
		writeJSON(w, http.StatusMethodNotAllowed, map[string]string{"error": "method not allowed"})
		return
	}
	if h.store == nil {
		writeJSON(w, http.StatusServiceUnavailable, map[string]string{
			"error": "store not initialized",
		})
		return
	}

	cid, _, err := readCIDRequest(r)
	if err != nil {
		writeJSON(w, http.StatusBadRequest, map[string]string{
			"error":   "invalid cid",
			"message": err.Error(),
		})
		return
	}

	if err := h.store.UnignorePin(r.Context(), cid); err != nil {
		writeJSON(w, http.StatusInternalServerError, map[string]string{
			"error":   "failed to unignore pin",
			"message": err.Error(),
		})
		return
	}

	writeJSON(w, http.StatusOK, map[string]interface{}{
		"success": true,
		"cid":     cid,
		"status":  "pending",
	})
}

// Uploads returns recent uploads.
func (h *Handlers) Uploads(w http.ResponseWriter, r *http.Request) {
	if h.store == nil {
		writeJSON(w, http.StatusServiceUnavailable, map[string]string{
			"error": "store not initialized",
		})
		return
	}

	limit, offset := parsePagination(r, 25, 200)
	uploads, err := h.store.ListUploads(r.Context(), limit, offset)
	if err != nil {
		writeJSON(w, http.StatusInternalServerError, map[string]string{
			"error":   "failed to list uploads",
			"message": err.Error(),
		})
		return
	}

	total, err := h.store.CountUploads(r.Context())
	if err != nil {
		writeJSON(w, http.StatusInternalServerError, map[string]string{
			"error":   "failed to count uploads",
			"message": err.Error(),
		})
		return
	}

	type uploadItem struct {
		store.UploadRecord
		URL         string `json:"url"`
		FallbackURL string `json:"fallback_url,omitempty"`
	}

	items := make([]uploadItem, 0, len(uploads))
	for _, upload := range uploads {
		filename := strings.TrimSpace(upload.Filename)
		if filename == "" {
			filename = "file"
			if upload.Extension != "" {
				filename += upload.Extension
			}
		}
		downloadURL := buildGatewayURL(h.cfg.GatewayURL, upload.CID, filename)
		fallbackURL := buildGatewayURL(h.cfg.GatewayFallbackURL, upload.CID, filename)
		items = append(items, uploadItem{
			UploadRecord: upload,
			URL:          downloadURL,
			FallbackURL:  fallbackURL,
		})
	}

	writeJSON(w, http.StatusOK, map[string]interface{}{
		"success": true,
		"uploads": items,
		"pagination": map[string]interface{}{
			"limit":   limit,
			"offset":  offset,
			"total":   total,
			"hasMore": offset+limit < total,
		},
	})
}

// Upload handles multipart uploads and stores metadata.
func (h *Handlers) Upload(w http.ResponseWriter, r *http.Request) {
	if h.ipfs == nil {
		writeJSON(w, http.StatusServiceUnavailable, map[string]string{
			"error":   "ipfs unavailable",
			"message": "ipfs client not initialized",
		})
		return
	}

	if r.Method == http.MethodPut {
		h.uploadRaw(w, r)
		return
	}

	if r.Method != http.MethodPost {
		writeJSON(w, http.StatusMethodNotAllowed, map[string]string{"error": "method not allowed"})
		return
	}

	r.Body = http.MaxBytesReader(w, r.Body, h.cfg.FileLimitBytes)

	if err := r.ParseMultipartForm(32 << 20); err != nil {
		writeJSON(w, http.StatusRequestEntityTooLarge, map[string]string{
			"error":   "file too large",
			"message": err.Error(),
		})
		return
	}
	if r.MultipartForm != nil {
		defer r.MultipartForm.RemoveAll()
	}

	file, header, err := r.FormFile("file")
	if err != nil {
		writeJSON(w, http.StatusBadRequest, map[string]string{"error": "missing file"})
		return
	}
	defer file.Close()

	filename := header.Filename
	ext := filepath.Ext(filename)
	mimeType := header.Header.Get("Content-Type")
	if mimeType == "" && ext != "" {
		mimeType = mime.TypeByExtension(ext)
	}
	if mimeType == "" {
		mimeType = "application/octet-stream"
	}

	expected := strings.TrimSpace(r.Header.Get("X-SHA-256"))
	hasher := sha256.New()
	var counter *readCounter
	var addReader io.Reader
	var shaSum string
	var sizeBytes int64

	if expected != "" {
		counter = &readCounter{r: file}
		if _, err := io.Copy(hasher, counter); err != nil {
			var maxErr *http.MaxBytesError
			if errors.As(err, &maxErr) {
				writeJSON(w, http.StatusRequestEntityTooLarge, map[string]string{
					"error":   "file too large",
					"message": err.Error(),
				})
				return
			}
			writeJSON(w, http.StatusBadRequest, map[string]string{
				"error":   "read failed",
				"message": err.Error(),
			})
			return
		}

		shaSum = hex.EncodeToString(hasher.Sum(nil))
		sizeBytes = counter.n
		if !strings.EqualFold(expected, shaSum) {
			writeJSON(w, http.StatusBadRequest, map[string]string{
				"error":   "sha256 mismatch",
				"message": "computed sha256 does not match X-SHA-256 header",
			})
			return
		}
		if _, err := file.Seek(0, io.SeekStart); err != nil {
			writeJSON(w, http.StatusBadRequest, map[string]string{
				"error":   "seek failed",
				"message": err.Error(),
			})
			return
		}
		addReader = file
	} else {
		counter = &readCounter{r: file}
		addReader = io.TeeReader(counter, hasher)
	}

	ctx := r.Context()
	addResult, err := h.ipfs.Add(ctx, addReader, h.cfg.PinUploads)
	if err != nil {
		writeJSON(w, http.StatusBadGateway, map[string]string{
			"error":   "ipfs upload failed",
			"message": err.Error(),
		})
		return
	}

	if expected == "" {
		shaSum = hex.EncodeToString(hasher.Sum(nil))
		sizeBytes = counter.n
	}

	if h.store != nil {
		storeErr := h.store.UpsertUpload(ctx, store.Upload{
			Sha256:    shaSum,
			CID:       addResult.Hash,
			Filename:  filename,
			Extension: ext,
			MimeType:  mimeType,
			SizeBytes: sizeBytes,
		})
		if storeErr != nil {
			writeJSON(w, http.StatusInternalServerError, map[string]string{
				"error":   "database error",
				"message": storeErr.Error(),
			})
			return
		}
	}

	gatewayURL := buildGatewayURL(h.cfg.GatewayURL, addResult.Hash, filename)
	fallbackURL := buildGatewayURL(h.cfg.GatewayFallbackURL, addResult.Hash, filename)

	response := map[string]interface{}{
		"status":   "success",
		"sha256":   shaSum,
		"cid":      addResult.Hash,
		"url":      gatewayURL,
		"fallback_url": fallbackURL,
		"size":     sizeBytes,
		"type":     mimeType,
		"filename": filename,
		"pinned":   h.cfg.PinUploads,
		"uploaded": time.Now().Unix(),
	}

	writeJSON(w, http.StatusOK, response)
}

func (h *Handlers) uploadRaw(w http.ResponseWriter, r *http.Request) {
	r.Body = http.MaxBytesReader(w, r.Body, h.cfg.FileLimitBytes)

	filename := headerFileName(r)
	ext := filepath.Ext(filename)
	mimeType := r.Header.Get("Content-Type")
	if mimeType == "" && ext != "" {
		mimeType = mime.TypeByExtension(ext)
	}
	if mimeType == "" {
		mimeType = "application/octet-stream"
	}

	expected := strings.TrimSpace(r.Header.Get("X-SHA-256"))
	hasher := sha256.New()
	var counter *readCounter
	var addReader io.Reader
	var shaSum string
	var sizeBytes int64
	var cleanup func()

	if expected != "" {
		tmp, err := os.CreateTemp("", "ipfstr-raw-*")
		if err != nil {
			writeJSON(w, http.StatusInternalServerError, map[string]string{
				"error":   "tempfile failed",
				"message": err.Error(),
			})
			return
		}
		cleanup = func() {
			_ = tmp.Close()
			_ = os.Remove(tmp.Name())
		}
		defer cleanup()

		counter = &readCounter{r: r.Body}
		if _, err := io.Copy(io.MultiWriter(tmp, hasher), counter); err != nil {
			var maxErr *http.MaxBytesError
			if errors.As(err, &maxErr) {
				writeJSON(w, http.StatusRequestEntityTooLarge, map[string]string{
					"error":   "file too large",
					"message": err.Error(),
				})
				return
			}
			writeJSON(w, http.StatusBadRequest, map[string]string{
				"error":   "read failed",
				"message": err.Error(),
			})
			return
		}

		shaSum = hex.EncodeToString(hasher.Sum(nil))
		sizeBytes = counter.n
		if !strings.EqualFold(expected, shaSum) {
			writeJSON(w, http.StatusBadRequest, map[string]string{
				"error":   "sha256 mismatch",
				"message": "computed sha256 does not match X-SHA-256 header",
			})
			return
		}
		if _, err := tmp.Seek(0, io.SeekStart); err != nil {
			writeJSON(w, http.StatusBadRequest, map[string]string{
				"error":   "seek failed",
				"message": err.Error(),
			})
			return
		}
		addReader = tmp
	} else {
		counter = &readCounter{r: r.Body}
		addReader = io.TeeReader(counter, hasher)
	}

	addResult, err := h.ipfs.Add(r.Context(), addReader, h.cfg.PinUploads)
	if err != nil {
		writeJSON(w, http.StatusBadGateway, map[string]string{
			"error":   "ipfs upload failed",
			"message": err.Error(),
		})
		return
	}

	if expected == "" {
		shaSum = hex.EncodeToString(hasher.Sum(nil))
		sizeBytes = counter.n
	}

	if h.store != nil {
		storeErr := h.store.UpsertUpload(r.Context(), store.Upload{
			Sha256:    shaSum,
			CID:       addResult.Hash,
			Filename:  filename,
			Extension: ext,
			MimeType:  mimeType,
			SizeBytes: sizeBytes,
		})
		if storeErr != nil {
			writeJSON(w, http.StatusInternalServerError, map[string]string{
				"error":   "database error",
				"message": storeErr.Error(),
			})
			return
		}
	}

	gatewayURL := buildGatewayURL(h.cfg.GatewayURL, addResult.Hash, filename)
	fallbackURL := buildGatewayURL(h.cfg.GatewayFallbackURL, addResult.Hash, filename)

	writeJSON(w, http.StatusOK, map[string]interface{}{
		"status":   "success",
		"sha256":   shaSum,
		"cid":      addResult.Hash,
		"url":      gatewayURL,
		"fallback_url": fallbackURL,
		"size":     sizeBytes,
		"type":     mimeType,
		"filename": filename,
		"pinned":   h.cfg.PinUploads,
		"uploaded": time.Now().Unix(),
	})
}

type cidRequest struct {
	CID       string `json:"cid"`
	Recursive *bool  `json:"recursive"`
}

func readCIDRequest(r *http.Request) (string, bool, error) {
	cid := strings.TrimSpace(r.URL.Query().Get("cid"))
	if cid == "" {
		cid = strings.TrimSpace(r.URL.Query().Get("arg"))
	}

	recursive := true
	if raw := strings.TrimSpace(r.URL.Query().Get("recursive")); raw != "" {
		if value, err := strconv.ParseBool(raw); err == nil {
			recursive = value
		} else {
			return "", recursive, fmt.Errorf("invalid recursive flag: %w", err)
		}
	}

	if cid != "" {
		return cid, recursive, nil
	}

	if r.Body == nil {
		return "", recursive, fmt.Errorf("cid is required")
	}

	var payload cidRequest
	if err := json.NewDecoder(r.Body).Decode(&payload); err != nil {
		return "", recursive, err
	}
	cid = strings.TrimSpace(payload.CID)
	if payload.Recursive != nil {
		recursive = *payload.Recursive
	}
	if cid == "" {
		return "", recursive, fmt.Errorf("cid is required")
	}
	return cid, recursive, nil
}

type addrRequest struct {
	Addr      string `json:"addr"`
	Multiaddr string `json:"multiaddr"`
}

func readAddrRequest(r *http.Request) (string, error) {
	addr := strings.TrimSpace(r.URL.Query().Get("addr"))
	if addr == "" {
		addr = strings.TrimSpace(r.URL.Query().Get("multiaddr"))
	}
	if addr != "" {
		return addr, nil
	}
	if r.Body == nil {
		return "", fmt.Errorf("address is required")
	}
	var payload addrRequest
	if err := json.NewDecoder(r.Body).Decode(&payload); err != nil {
		return "", err
	}
	addr = strings.TrimSpace(payload.Addr)
	if addr == "" {
		addr = strings.TrimSpace(payload.Multiaddr)
	}
	if addr == "" {
		return "", fmt.Errorf("address is required")
	}
	return addr, nil
}

func headerFileName(r *http.Request) string {
	candidates := []string{
		r.Header.Get("X-File-Name"),
		r.Header.Get("X-Filename"),
		r.Header.Get("X-Content-Name"),
		r.URL.Query().Get("filename"),
	}
	for _, name := range candidates {
		if strings.TrimSpace(name) != "" {
			return name
		}
	}

	if r.Header.Get("Content-Type") != "" {
		if exts, _ := mime.ExtensionsByType(r.Header.Get("Content-Type")); len(exts) > 0 {
			return "upload" + exts[0]
		}
	}

	return "upload.bin"
}

func parsePagination(r *http.Request, defaultLimit int, maxLimit int) (int, int) {
	limit := defaultLimit
	offset := 0

	if raw := r.URL.Query().Get("limit"); raw != "" {
		if value, err := strconv.Atoi(raw); err == nil {
			limit = value
		}
	}
	if raw := r.URL.Query().Get("offset"); raw != "" {
		if value, err := strconv.Atoi(raw); err == nil {
			offset = value
		}
	}

	if limit <= 0 {
		limit = defaultLimit
	}
	if limit > maxLimit {
		limit = maxLimit
	}
	if offset < 0 {
		offset = 0
	}

	return limit, offset
}

func writeJSON(w http.ResponseWriter, status int, payload interface{}) {
	w.Header().Set("Content-Type", "application/json")
	w.WriteHeader(status)
	_ = json.NewEncoder(w).Encode(payload)
}

type readCounter struct {
	r io.Reader
	n int64
}

func (rc *readCounter) Read(p []byte) (int, error) {
	n, err := rc.r.Read(p)
	rc.n += int64(n)
	return n, err
}

func buildGatewayURL(baseURL string, cid string, filename string) string {
	if baseURL == "" || cid == "" {
		return ""
	}
	urlValue := fmt.Sprintf("%s%s", baseURL, cid)
	if strings.TrimSpace(filename) == "" {
		return urlValue
	}
	return urlValue + "?filename=" + url.QueryEscape(filename)
}
