package ipfs

import (
	"bytes"
	"context"
	"encoding/json"
	"fmt"
	"io"
	"mime/multipart"
	"net/http"
	"net/url"
	"strconv"
	"strings"
	"time"

	shell "github.com/ipfs/go-ipfs-api"
)

// Client wraps the IPFS HTTP API.
type Client struct {
	shell   *shell.Shell
	apiURL  string
	http    *http.Client
	pinHTTP *http.Client
}

// AddResult matches the IPFS add response.
type AddResult struct {
	Hash string `json:"Hash"`
	Name string `json:"Name"`
	Size string `json:"Size"`
}

// PinInfo describes a pinned CID.
type PinInfo struct {
	CID  string `json:"cid"`
	Type string `json:"type,omitempty"`
}

// SwarmPeer describes a connected peer.
type SwarmPeer struct {
	Peer string `json:"peer"`
	Addr string `json:"addr"`
}

// NodeStats describes IPFS daemon status.
type NodeStats struct {
	Available       bool     `json:"available"`
	Version         string   `json:"version"`
	PeerID          string   `json:"peer_id"`
	AgentVersion    string   `json:"agent_version"`
	ProtocolVersion string   `json:"protocol_version"`
	Addresses       []string `json:"addresses,omitempty"`
	RepoSizeBytes   int64    `json:"repo_size_bytes"`
	StorageMaxBytes int64    `json:"storage_max_bytes"`
	NumObjects      int64    `json:"num_objects"`
	RepoPath        string   `json:"repo_path"`
	PeerCount       int      `json:"peer_count"`
	PinnedCount     int      `json:"pinned_count"`
	Error           string   `json:"error,omitempty"`
}

// New creates an IPFS client.
func New(apiURL string, timeout time.Duration, pinTimeout time.Duration) *Client {
	trimmed := strings.TrimRight(apiURL, "/")
	apiURLWithScheme := trimmed
	if !strings.Contains(apiURLWithScheme, "://") {
		apiURLWithScheme = "http://" + apiURLWithScheme
	}
	if timeout <= 0 {
		timeout = 30 * time.Second
	}
	if pinTimeout <= 0 {
		pinTimeout = timeout
	}

	shellAddr := trimmed
	if strings.Contains(trimmed, "://") {
		if parsed, err := url.Parse(trimmed); err == nil && parsed.Host != "" {
			shellAddr = parsed.Host
		}
	}
	return &Client{
		shell:  shell.NewShell(shellAddr),
		apiURL: apiURLWithScheme,
		http: &http.Client{
			Timeout: timeout,
		},
		pinHTTP: &http.Client{
			Timeout: pinTimeout,
		},
	}
}

// IsUp checks IPFS availability.
func (c *Client) IsUp() bool {
	if c == nil {
		return false
	}
	ctx, cancel := context.WithTimeout(context.Background(), 3*time.Second)
	defer cancel()
	endpoint, err := c.buildURL("version", nil)
	if err != nil {
		return false
	}
	_, err = c.doPost(ctx, endpoint)
	return err == nil
}

// Add adds content to IPFS without pinning.
func (c *Client) Add(ctx context.Context, reader io.Reader, pin bool) (AddResult, error) {
	if c == nil {
		return AddResult{}, fmt.Errorf("ipfs client not initialized")
	}

	pinValue := "false"
	if pin {
		pinValue = "true"
	}
	endpoint, err := c.buildURL("add", url.Values{
		"pin": []string{pinValue},
	})
	if err != nil {
		return AddResult{}, err
	}

	bodyReader, contentType := buildMultipart(reader)
	req, err := http.NewRequestWithContext(ctx, http.MethodPost, endpoint, bodyReader)
	if err != nil {
		return AddResult{}, err
	}
	req.Header.Set("Content-Type", contentType)

	resp, err := c.http.Do(req)
	if err != nil {
		return AddResult{}, fmt.Errorf("ipfs add request: %w", err)
	}
	defer resp.Body.Close()

	respBody, err := io.ReadAll(resp.Body)
	if err != nil {
		return AddResult{}, fmt.Errorf("read ipfs add response: %w", err)
	}
	if resp.StatusCode >= 300 {
		return AddResult{}, fmt.Errorf("ipfs add response: %s", strings.TrimSpace(string(respBody)))
	}

	var result AddResult
	for _, line := range bytes.Split(respBody, []byte("\n")) {
		line = bytes.TrimSpace(line)
		if len(line) == 0 {
			continue
		}
		if err := json.Unmarshal(line, &result); err == nil && result.Hash != "" {
			return result, nil
		}
	}

	return AddResult{}, fmt.Errorf("ipfs returned empty hash")
}

// Pin pins a CID in IPFS.
func (c *Client) Pin(ctx context.Context, cid string) error {
	if c == nil {
		return fmt.Errorf("ipfs client not initialized")
	}

	return c.PinWithOptions(ctx, cid, true)
}

// PinWithOptions pins a CID with a recursive flag.
func (c *Client) PinWithOptions(ctx context.Context, cid string, recursive bool) error {
	if c == nil {
		return fmt.Errorf("ipfs client not initialized")
	}

	endpoint, err := c.buildURL("pin/add", url.Values{
		"arg":       []string{cid},
		"recursive": []string{strconv.FormatBool(recursive)},
	})
	if err != nil {
		return err
	}
	_, err = c.doPostWithClient(ctx, endpoint, c.pinHTTP)
	return err
}

// Unpin removes a pinned CID.
func (c *Client) Unpin(ctx context.Context, cid string, recursive bool) error {
	if c == nil {
		return fmt.Errorf("ipfs client not initialized")
	}

	endpoint, err := c.buildURL("pin/rm", url.Values{
		"arg":       []string{cid},
		"recursive": []string{strconv.FormatBool(recursive)},
	})
	if err != nil {
		return err
	}
	_, err = c.doPostWithClient(ctx, endpoint, c.pinHTTP)
	return err
}

// PinLs lists pinned CIDs with their type.
func (c *Client) PinLs(ctx context.Context, pinType string, cid string) ([]PinInfo, error) {
	if c == nil {
		return nil, fmt.Errorf("ipfs client not initialized")
	}

	params := url.Values{}
	if strings.TrimSpace(pinType) != "" {
		params.Set("type", pinType)
	}
	if strings.TrimSpace(cid) != "" {
		params.Set("arg", cid)
	}
	endpoint, err := c.buildURL("pin/ls", params)
	if err != nil {
		return nil, err
	}
	body, err := c.doPost(ctx, endpoint)
	if err != nil {
		if strings.Contains(err.Error(), "not pinned") {
			return nil, nil
		}
		return nil, err
	}

	var payload struct {
		Keys map[string]struct {
			Type string `json:"Type"`
		} `json:"Keys"`
	}
	if err := json.Unmarshal(body, &payload); err != nil {
		return nil, fmt.Errorf("decode pin ls: %w", err)
	}

	pins := make([]PinInfo, 0, len(payload.Keys))
	for key, entry := range payload.Keys {
		pins = append(pins, PinInfo{CID: key, Type: entry.Type})
	}
	return pins, nil
}

// IsPinned returns true if a CID is already pinned.
func (c *Client) IsPinned(ctx context.Context, cid string) (bool, error) {
	if c == nil {
		return false, fmt.Errorf("ipfs client not initialized")
	}

	endpoint, err := c.buildURL("pin/ls", url.Values{
		"arg":  []string{cid},
		"type": []string{"recursive"},
	})
	if err != nil {
		return false, err
	}

	body, err := c.doPost(ctx, endpoint)
	if err != nil {
		if strings.Contains(err.Error(), "not pinned") {
			return false, nil
		}
		return false, err
	}

	var payload struct {
		Keys map[string]interface{} `json:"Keys"`
	}
	if err := json.Unmarshal(body, &payload); err != nil {
		return false, fmt.Errorf("decode pin ls: %w", err)
	}

	return len(payload.Keys) > 0, nil
}

// CIDSize returns the size of a CID using files/stat or block/stat.
func (c *Client) CIDSize(ctx context.Context, cid string) (int64, error) {
	if c == nil {
		return 0, fmt.Errorf("ipfs client not initialized")
	}

	var stat struct {
		CumulativeSize int64 `json:"CumulativeSize"`
		Size           int64 `json:"Size"`
	}

	filesURL, err := c.buildURL("files/stat", url.Values{
		"arg": []string{"/ipfs/" + cid},
	})
	if err != nil {
		return 0, err
	}
	body, err := c.doPost(ctx, filesURL)
	if err == nil {
		if jsonErr := json.Unmarshal(body, &stat); jsonErr == nil {
			if stat.CumulativeSize > 0 {
				return stat.CumulativeSize, nil
			}
			if stat.Size > 0 {
				return stat.Size, nil
			}
		}
	}

	blockURL, err := c.buildURL("block/stat", url.Values{
		"arg": []string{cid},
	})
	if err != nil {
		return 0, err
	}
	body, err = c.doPost(ctx, blockURL)
	if err != nil {
		return 0, fmt.Errorf("ipfs block stat: %w", err)
	}

	if err := json.Unmarshal(body, &stat); err != nil {
		return 0, fmt.Errorf("decode block stat: %w", err)
	}

	if stat.Size > 0 {
		return stat.Size, nil
	}

	return stat.CumulativeSize, nil
}

// Cat returns a reader for a CID's content.
func (c *Client) Cat(ctx context.Context, cid string) (io.ReadCloser, error) {
	if c == nil {
		return nil, fmt.Errorf("ipfs client not initialized")
	}
	endpoint, err := c.buildURL("cat", url.Values{
		"arg": []string{cid},
	})
	if err != nil {
		return nil, err
	}
	req, err := http.NewRequestWithContext(ctx, http.MethodPost, endpoint, nil)
	if err != nil {
		return nil, err
	}
	resp, err := c.http.Do(req)
	if err != nil {
		return nil, fmt.Errorf("ipfs cat request: %w", err)
	}

	if resp.StatusCode >= 300 {
		body, _ := io.ReadAll(resp.Body)
		resp.Body.Close()
		return nil, fmt.Errorf("ipfs cat response: %s", strings.TrimSpace(string(body)))
	}

	return resp.Body, nil
}

// NodeInfo gathers useful IPFS daemon stats.
func (c *Client) NodeInfo(ctx context.Context) (NodeStats, error) {
	if c == nil {
		return NodeStats{}, fmt.Errorf("ipfs client not initialized")
	}

	stats := NodeStats{}
	var errors []string

	if version, err := c.version(ctx); err != nil {
		errors = append(errors, "version: "+err.Error())
	} else {
		stats.Version = version
	}

	idInfo, err := c.id(ctx)
	if err != nil {
		errors = append(errors, "id: "+err.Error())
	} else {
		stats.PeerID = idInfo.ID
		stats.AgentVersion = idInfo.AgentVersion
		stats.ProtocolVersion = idInfo.ProtocolVersion
		stats.Addresses = idInfo.Addresses
	}

	repo, err := c.repoStat(ctx)
	if err != nil {
		errors = append(errors, "repo: "+err.Error())
	} else {
		stats.RepoSizeBytes = repo.RepoSizeBytes
		stats.StorageMaxBytes = repo.StorageMaxBytes
		stats.NumObjects = repo.NumObjects
		stats.RepoPath = repo.RepoPath
	}

	peerCount, err := c.swarmPeerCount(ctx)
	if err != nil {
		errors = append(errors, "peers: "+err.Error())
	} else {
		stats.PeerCount = peerCount
	}

	pinnedCount, err := c.pinCount(ctx)
	if err != nil {
		errors = append(errors, "pins: "+err.Error())
	} else {
		stats.PinnedCount = pinnedCount
	}

	if len(errors) > 0 {
		return stats, fmt.Errorf(strings.Join(errors, "; "))
	}

	return stats, nil
}

func (c *Client) buildURL(path string, params url.Values) (string, error) {
	if c == nil || c.apiURL == "" {
		return "", fmt.Errorf("ipfs client not initialized")
	}
	endpoint := fmt.Sprintf("%s/api/v0/%s", c.apiURL, path)
	if params != nil {
		endpoint = endpoint + "?" + params.Encode()
	}
	return endpoint, nil
}

type idResponse struct {
	ID              string   `json:"ID"`
	Addresses       []string `json:"Addresses"`
	AgentVersion    string   `json:"AgentVersion"`
	ProtocolVersion string   `json:"ProtocolVersion"`
}

type repoStatResponse struct {
	RepoSize   json.RawMessage `json:"RepoSize"`
	StorageMax json.RawMessage `json:"StorageMax"`
	NumObjects json.RawMessage `json:"NumObjects"`
	RepoPath   string          `json:"RepoPath"`
}

type repoStat struct {
	RepoSizeBytes   int64
	StorageMaxBytes int64
	NumObjects      int64
	RepoPath        string
}

func (c *Client) version(ctx context.Context) (string, error) {
	endpoint, err := c.buildURL("version", nil)
	if err != nil {
		return "", err
	}
	body, err := c.doPost(ctx, endpoint)
	if err != nil {
		return "", err
	}
	var payload struct {
		Version string `json:"Version"`
	}
	if err := json.Unmarshal(body, &payload); err != nil {
		return "", fmt.Errorf("decode version: %w", err)
	}
	return payload.Version, nil
}

func (c *Client) id(ctx context.Context) (idResponse, error) {
	endpoint, err := c.buildURL("id", nil)
	if err != nil {
		return idResponse{}, err
	}
	body, err := c.doPost(ctx, endpoint)
	if err != nil {
		return idResponse{}, err
	}
	var payload idResponse
	if err := json.Unmarshal(body, &payload); err != nil {
		return idResponse{}, fmt.Errorf("decode id: %w", err)
	}
	return payload, nil
}

func (c *Client) repoStat(ctx context.Context) (repoStat, error) {
	endpoint, err := c.buildURL("repo/stat", nil)
	if err != nil {
		return repoStat{}, err
	}
	body, err := c.doPost(ctx, endpoint)
	if err != nil {
		return repoStat{}, err
	}
	var payload repoStatResponse
	if err := json.Unmarshal(body, &payload); err != nil {
		return repoStat{}, fmt.Errorf("decode repo stat: %w", err)
	}
	repoSize, err := parseJSONInt(payload.RepoSize)
	if err != nil {
		return repoStat{}, fmt.Errorf("parse repo size: %w", err)
	}
	storageMax, err := parseJSONInt(payload.StorageMax)
	if err != nil {
		return repoStat{}, fmt.Errorf("parse storage max: %w", err)
	}
	numObjects, err := parseJSONInt(payload.NumObjects)
	if err != nil {
		return repoStat{}, fmt.Errorf("parse num objects: %w", err)
	}
	return repoStat{
		RepoSizeBytes:   repoSize,
		StorageMaxBytes: storageMax,
		NumObjects:      numObjects,
		RepoPath:        payload.RepoPath,
	}, nil
}

func (c *Client) swarmPeerCount(ctx context.Context) (int, error) {
	endpoint, err := c.buildURL("swarm/peers", nil)
	if err != nil {
		return 0, err
	}
	body, err := c.doPost(ctx, endpoint)
	if err != nil {
		return 0, err
	}
	var payload struct {
		Peers []interface{} `json:"Peers"`
	}
	if err := json.Unmarshal(body, &payload); err != nil {
		return 0, fmt.Errorf("decode swarm peers: %w", err)
	}
	return len(payload.Peers), nil
}

// SwarmPeers returns connected peers.
func (c *Client) SwarmPeers(ctx context.Context) ([]SwarmPeer, error) {
	if c == nil {
		return nil, fmt.Errorf("ipfs client not initialized")
	}

	endpoint, err := c.buildURL("swarm/peers", nil)
	if err != nil {
		return nil, err
	}
	body, err := c.doPost(ctx, endpoint)
	if err != nil {
		return nil, err
	}
	var payload struct {
		Peers []struct {
			Peer string `json:"Peer"`
			Addr string `json:"Addr"`
		} `json:"Peers"`
	}
	if err := json.Unmarshal(body, &payload); err != nil {
		return nil, fmt.Errorf("decode swarm peers: %w", err)
	}

	peers := make([]SwarmPeer, 0, len(payload.Peers))
	for _, peer := range payload.Peers {
		peers = append(peers, SwarmPeer{Peer: peer.Peer, Addr: peer.Addr})
	}
	return peers, nil
}

// SwarmConnect connects to a multiaddr.
func (c *Client) SwarmConnect(ctx context.Context, addr string) ([]string, error) {
	if c == nil {
		return nil, fmt.Errorf("ipfs client not initialized")
	}

	endpoint, err := c.buildURL("swarm/connect", url.Values{
		"arg": []string{addr},
	})
	if err != nil {
		return nil, err
	}
	body, err := c.doPost(ctx, endpoint)
	if err != nil {
		return nil, err
	}
	var payload struct {
		Strings []string `json:"Strings"`
	}
	if err := json.Unmarshal(body, &payload); err != nil {
		return nil, fmt.Errorf("decode swarm connect: %w", err)
	}
	return payload.Strings, nil
}

// SwarmDisconnect disconnects from a multiaddr.
func (c *Client) SwarmDisconnect(ctx context.Context, addr string) ([]string, error) {
	if c == nil {
		return nil, fmt.Errorf("ipfs client not initialized")
	}

	endpoint, err := c.buildURL("swarm/disconnect", url.Values{
		"arg": []string{addr},
	})
	if err != nil {
		return nil, err
	}
	body, err := c.doPost(ctx, endpoint)
	if err != nil {
		return nil, err
	}
	var payload struct {
		Strings []string `json:"Strings"`
	}
	if err := json.Unmarshal(body, &payload); err != nil {
		return nil, fmt.Errorf("decode swarm disconnect: %w", err)
	}
	return payload.Strings, nil
}

func (c *Client) pinCount(ctx context.Context) (int, error) {
	endpoint, err := c.buildURL("pin/ls", url.Values{
		"type": []string{"recursive"},
	})
	if err != nil {
		return 0, err
	}
	body, err := c.doPost(ctx, endpoint)
	if err != nil {
		return 0, err
	}
	var payload struct {
		Keys map[string]interface{} `json:"Keys"`
	}
	if err := json.Unmarshal(body, &payload); err != nil {
		return 0, fmt.Errorf("decode pin ls: %w", err)
	}
	return len(payload.Keys), nil
}

// RepoGC triggers IPFS garbage collection and returns the number of removed keys.
func (c *Client) RepoGC(ctx context.Context) (int, error) {
	endpoint, err := c.buildURL("repo/gc", nil)
	if err != nil {
		return 0, err
	}
	req, err := http.NewRequestWithContext(ctx, http.MethodPost, endpoint, nil)
	if err != nil {
		return 0, err
	}
	resp, err := c.http.Do(req)
	if err != nil {
		return 0, err
	}
	defer resp.Body.Close()

	if resp.StatusCode >= 300 {
		body, _ := io.ReadAll(resp.Body)
		return 0, fmt.Errorf("repo gc failed: %s", strings.TrimSpace(string(body)))
	}

	decoder := json.NewDecoder(resp.Body)
	removed := 0
	for {
		var entry map[string]interface{}
		if err := decoder.Decode(&entry); err != nil {
			if err == io.EOF {
				break
			}
			return removed, fmt.Errorf("decode gc entry: %w", err)
		}
		if _, ok := entry["Key"]; ok {
			removed++
		}
	}
	return removed, nil
}

func parseJSONInt(raw json.RawMessage) (int64, error) {
	if len(raw) == 0 {
		return 0, nil
	}
	var value int64
	if err := json.Unmarshal(raw, &value); err == nil {
		return value, nil
	}
	var asString string
	if err := json.Unmarshal(raw, &asString); err == nil {
		if asString == "" {
			return 0, nil
		}
		parsed, err := strconv.ParseInt(asString, 10, 64)
		if err != nil {
			return 0, err
		}
		return parsed, nil
	}
	var asFloat float64
	if err := json.Unmarshal(raw, &asFloat); err == nil {
		return int64(asFloat), nil
	}
	return 0, fmt.Errorf("unsupported number format")
}

func (c *Client) doPost(ctx context.Context, endpoint string) ([]byte, error) {
	return c.doPostWithClient(ctx, endpoint, c.http)
}

func (c *Client) doPostWithClient(ctx context.Context, endpoint string, client *http.Client) ([]byte, error) {
	req, err := http.NewRequestWithContext(ctx, http.MethodPost, endpoint, nil)
	if err != nil {
		return nil, err
	}
	if client == nil {
		client = c.http
	}
	resp, err := client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()

	body, err := io.ReadAll(resp.Body)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode >= 300 {
		return nil, fmt.Errorf("ipfs request failed: %s", strings.TrimSpace(string(body)))
	}

	return body, nil
}

func buildMultipart(reader io.Reader) (io.Reader, string) {
	pr, pw := io.Pipe()
	writer := multipart.NewWriter(pw)

	go func() {
		defer pw.Close()
		defer writer.Close()

		part, err := writer.CreateFormFile("file", "upload")
		if err != nil {
			_ = pw.CloseWithError(err)
			return
		}
		if _, err := io.Copy(part, reader); err != nil {
			_ = pw.CloseWithError(err)
			return
		}
	}()

	return pr, writer.FormDataContentType()
}
