package nostrworker

import (
	"context"
	"database/sql"
	"encoding/hex"
	"errors"
	"fmt"
	"log"
	"math/rand"
	"reflect"
	"regexp"
	"strconv"
	"strings"
	"time"

	"ipfstr/internal/config"
	"ipfstr/internal/ipfs"
	"ipfstr/internal/store"

	"github.com/nbd-wtf/go-nostr"
	"github.com/nbd-wtf/go-nostr/nip19"
)

const (
	stateLastRun   = "nostr_last_run"
	stateLastError = "nostr_last_error"
)

var (
	cidV0Pattern   = regexp.MustCompile(`\bQm[1-9A-HJ-NP-Za-km-z]{44}\b`)
	cidV1Pattern   = regexp.MustCompile(`\b[bB][a-zA-Z2-7]{58,}\b`)
	ipfsURLPattern = regexp.MustCompile(`(?i)\b(?:ipfs://|https?://(?:[^/]+\.)?(?:dweb\.link|ipfs\.io)/ipfs/)([A-Za-z0-9._-]+)`)
)

// Worker manages Nostr discovery and pinning.
type Worker struct {
	cfg    config.Config
	store  *store.Store
	ipfs   *ipfs.Client
	logger *log.Logger
	rng    *rand.Rand
	pubkey string
	relays []string
}

// New creates a new worker.
func New(cfg config.Config, storeConn *store.Store, ipfsClient *ipfs.Client, logger *log.Logger) (*Worker, error) {
	pubkey, err := decodePubkey(cfg.NostrPubkey)
	if err != nil {
		return nil, err
	}
	if logger == nil {
		logger = log.Default()
	}

	relays := cfg.NostrRelays
	if len(relays) == 0 {
		relays = defaultRelays()
	}

	return &Worker{
		cfg:    cfg,
		store:  storeConn,
		ipfs:   ipfsClient,
		logger: logger,
		rng:    rand.New(rand.NewSource(time.Now().UnixNano())),
		pubkey: pubkey,
		relays: relays,
	}, nil
}

// Enabled reports if Nostr is configured.
func (w *Worker) Enabled() bool {
	return w != nil && w.pubkey != ""
}

// Start runs discovery and pinning loops.
func (w *Worker) Start(ctx context.Context) {
	if !w.Enabled() {
		return
	}

	if w.store != nil {
		reset, err := w.store.ResetInProgressPins(ctx)
		if err != nil {
			w.logger.Printf("[nostr] reset in_progress failed: %v", err)
		} else if reset > 0 {
			w.logger.Printf("[nostr] reset %d in_progress pins", reset)
		}
	}

	go w.discoveryLoop(ctx)
	go w.pinnerLoop(ctx)
}

func (w *Worker) discoveryLoop(ctx context.Context) {
	w.runDiscovery(ctx)
	ticker := time.NewTicker(w.cfg.NostrDiscoveryEvery)
	defer ticker.Stop()

	for {
		select {
		case <-ctx.Done():
			return
		case <-ticker.C:
			w.runDiscovery(ctx)
		}
	}
}

func (w *Worker) runDiscovery(ctx context.Context) {
	if w.store == nil {
		return
	}

	lastRunUnix := w.lastRun(ctx)
	if lastRunUnix == 0 {
		lastRunUnix = time.Now().Add(-w.cfg.NostrLookback).Unix()
	}

	w.logger.Printf("[nostr] discovery start since=%d", lastRunUnix)
	since := nostr.Timestamp(lastRunUnix)

	events, err := w.fetchEvents(ctx, []int{1, 6, 30023, 30024, 9802}, &since)
	if err != nil {
		w.logger.Printf("[nostr] discovery error: %v", err)
		_ = w.store.SetState(ctx, stateLastError, err.Error())
		return
	}

	deleteEvents, err := w.fetchEvents(ctx, []int{5}, &since)
	if err != nil {
		w.logger.Printf("[nostr] delete fetch error: %v", err)
	}

	deleted := deletedIDs(deleteEvents)

	cidMap := make(map[string]store.Pin)
	for _, evt := range events {
		if _, ok := deleted[evt.ID]; ok {
			continue
		}
		if isExpired(evt) {
			continue
		}
		for _, cid := range extractEventCIDs(evt) {
			if _, exists := cidMap[cid]; !exists {
				cidMap[cid] = store.Pin{
					CID:     cid,
					EventID: evt.ID,
					Author:  evt.PubKey,
					Type:    "self",
					Status:  "pending",
				}
			}
		}
	}

	pins := make([]store.Pin, 0, len(cidMap))
	for _, pin := range cidMap {
		pins = append(pins, pin)
	}

	inserted, err := w.store.InsertPins(ctx, pins)
	if err != nil {
		w.logger.Printf("[nostr] insert error: %v", err)
		_ = w.store.SetState(ctx, stateLastError, err.Error())
		return
	}

	now := time.Now().Unix()
	_ = w.store.SetState(ctx, stateLastRun, strconv.FormatInt(now, 10))
	_ = w.store.SetState(ctx, stateLastError, "")

	w.logger.Printf("[nostr] discovery done events=%d cids=%d inserted=%d", len(events), len(pins), inserted)
}

func (w *Worker) pinnerLoop(ctx context.Context) {
	if w.store == nil || w.ipfs == nil {
		return
	}

	for {
		select {
		case <-ctx.Done():
			return
		default:
		}

		pin, err := w.store.GetRandomPendingPin(ctx)
		if err != nil {
			if errors.Is(err, context.Canceled) {
				return
			}
			if errors.Is(err, sql.ErrNoRows) {
				w.sleep(ctx, 44*time.Second)
				continue
			}
			w.sleep(ctx, 44*time.Second)
			continue
		}

		_ = w.store.SetPinInProgress(ctx, pin.CID, true)

		pinned, err := w.ipfs.IsPinned(ctx, pin.CID)
		if err == nil && pinned {
			size, _ := w.ipfs.CIDSize(ctx, pin.CID)
			_ = w.store.UpdatePinResult(ctx, pin.CID, "pinned", size, "")
			w.sleep(ctx, w.randomDelay())
			continue
		}

		err = w.ipfs.Pin(ctx, pin.CID)
		if err != nil {
			_ = w.store.UpdatePinResult(ctx, pin.CID, "failed", 0, err.Error())
			w.logger.Printf("[nostr] pin failed cid=%s error=%v", pin.CID, err)
			w.sleep(ctx, w.randomDelay())
			continue
		}

		size, _ := w.ipfs.CIDSize(ctx, pin.CID)
		_ = w.store.UpdatePinResult(ctx, pin.CID, "pinned", size, "")
		w.logger.Printf("[nostr] pinned cid=%s size=%d", pin.CID, size)
		w.sleep(ctx, w.randomDelay())
	}
}

func (w *Worker) randomDelay() time.Duration {
	min := w.cfg.NostrPinnerMinDelay
	max := w.cfg.NostrPinnerMaxDelay
	if max <= min {
		return min
	}
	gap := max - min
	return min + time.Duration(w.rng.Int63n(int64(gap)))
}

func (w *Worker) sleep(ctx context.Context, delay time.Duration) {
	timer := time.NewTimer(delay)
	defer timer.Stop()
	select {
	case <-ctx.Done():
	case <-timer.C:
	}
}

func (w *Worker) fetchEvents(ctx context.Context, kinds []int, since *nostr.Timestamp) ([]*nostr.Event, error) {
	timeoutCtx, cancel := context.WithTimeout(ctx, 20*time.Second)
	defer cancel()

	pool := nostr.NewSimplePool(timeoutCtx)
	filter := nostr.Filter{
		Authors: []string{w.pubkey},
		Kinds:   kinds,
		Limit:   500,
	}
	if since != nil {
		filter.Since = since
	}

	events, err := queryPool(timeoutCtx, pool, w.relays, filter)
	if err == nil {
		return events, nil
	}

	w.logger.Printf("[nostr] pool query failed, falling back to relay connect: %v", err)
	return w.fetchEventsDirect(timeoutCtx, filter)
}

func (w *Worker) fetchEventsDirect(ctx context.Context, filter nostr.Filter) ([]*nostr.Event, error) {
	unique := make(map[string]*nostr.Event)

	for _, relayURL := range w.relays {
		relayCtx, cancel := context.WithTimeout(ctx, 15*time.Second)
		relay, err := nostr.RelayConnect(relayCtx, relayURL)
		if err != nil {
			cancel()
			w.logger.Printf("[nostr] relay connect failed url=%s error=%v", relayURL, err)
			continue
		}

		sub, err := subscribeRelay(relayCtx, relay, filter)
		if err != nil {
			relay.Close()
			cancel()
			w.logger.Printf("[nostr] relay subscribe failed url=%s error=%v", relayURL, err)
			continue
		}

		events, err := collectFromSubscription(relayCtx, reflect.ValueOf(sub))
		unsubscribe(reflect.ValueOf(sub))
		relay.Close()
		cancel()
		if err != nil {
			w.logger.Printf("[nostr] relay subscription error url=%s error=%v", relayURL, err)
			continue
		}

		for _, evt := range events {
			if evt == nil || evt.ID == "" {
				continue
			}
			unique[evt.ID] = evt
		}
	}

	results := make([]*nostr.Event, 0, len(unique))
	for _, evt := range unique {
		results = append(results, evt)
	}
	return results, nil
}

func (w *Worker) lastRun(ctx context.Context) int64 {
	value, err := w.store.GetState(ctx, stateLastRun)
	if err != nil || value == "" {
		return 0
	}
	parsed, err := strconv.ParseInt(value, 10, 64)
	if err != nil {
		return 0
	}
	return parsed
}

func extractCIDs(text string) []string {
	found := make(map[string]struct{})

	for _, match := range ipfsURLPattern.FindAllStringSubmatch(text, -1) {
		cid := cleanCID(match[1])
		if cid != "" {
			found[cid] = struct{}{}
		}
	}

	for _, cid := range cidV0Pattern.FindAllString(text, -1) {
		clean := cleanCID(cid)
		if clean != "" {
			found[clean] = struct{}{}
		}
	}

	for _, cid := range cidV1Pattern.FindAllString(text, -1) {
		clean := cleanCID(cid)
		if clean != "" {
			found[clean] = struct{}{}
		}
	}

	results := make([]string, 0, len(found))
	for cid := range found {
		results = append(results, cid)
	}

	return results
}

func extractEventCIDs(evt *nostr.Event) []string {
	if evt == nil {
		return nil
	}

	found := make(map[string]struct{})
	for _, cid := range extractCIDs(evt.Content) {
		found[cid] = struct{}{}
	}

	for _, tag := range evt.Tags {
		for _, value := range tag {
			for _, cid := range extractCIDs(value) {
				found[cid] = struct{}{}
			}
		}
	}

	if len(found) == 0 {
		return nil
	}

	results := make([]string, 0, len(found))
	for cid := range found {
		results = append(results, cid)
	}
	return results
}

func cleanCID(raw string) string {
	if raw == "" {
		return ""
	}
	trimmed := strings.SplitN(raw, "?", 2)[0]
	trimmed = strings.SplitN(trimmed, "#", 2)[0]
	trimmed = strings.SplitN(trimmed, "/", 2)[0]
	trimmed = strings.Map(func(r rune) rune {
		if (r >= 'a' && r <= 'z') || (r >= 'A' && r <= 'Z') || (r >= '0' && r <= '9') {
			return r
		}
		return -1
	}, trimmed)
	if len(trimmed) < 46 || len(trimmed) > 120 {
		return ""
	}
	return trimmed
}

func isExpired(evt *nostr.Event) bool {
	for _, tag := range evt.Tags {
		if len(tag) >= 2 && tag[0] == "expiration" {
			value, err := strconv.ParseInt(tag[1], 10, 64)
			if err == nil && time.Now().Unix() > value {
				return true
			}
		}
	}
	return false
}

func deletedIDs(events []*nostr.Event) map[string]struct{} {
	deleted := make(map[string]struct{})
	for _, evt := range events {
		for _, tag := range evt.Tags {
			if len(tag) >= 2 && tag[0] == "e" {
				deleted[tag[1]] = struct{}{}
			}
		}
	}
	return deleted
}

func decodePubkey(input string) (string, error) {
	candidate := strings.TrimSpace(input)
	if candidate == "" {
		return "", nil
	}

	if isHex64(candidate) {
		return strings.ToLower(candidate), nil
	}

	prefix, data, err := nip19.Decode(candidate)
	if err != nil {
		return "", err
	}
	if prefix != "npub" {
		return "", fmt.Errorf("unsupported nostr pubkey: %s", prefix)
	}
	pubkey, ok := data.(string)
	if !ok {
		return "", fmt.Errorf("invalid npub payload")
	}
	return pubkey, nil
}

func isHex64(value string) bool {
	if len(value) != 64 {
		return false
	}
	_, err := hex.DecodeString(value)
	return err == nil
}

func defaultRelays() []string {
	return []string{
		"wss://auth.nostr1.com",
		"wss://bostr.bitcointxoko.com",
		"wss://eden.nostr.land",
		"wss://groups.0xchat.com",
		"wss://inbox.nostr.wine",
		"wss://nos.lol",
		"wss://nostr.band",
		"wss://nostr.data.haus",
		"wss://nostr.mom",
		"wss://nostr.oxtr.dev",
		"wss://relay.damus.io",
		"wss://relay.nostr.band",
		"wss://relay.nostr.bg",
		"wss://relay.primal.net",
		"wss://relay.siamstr.com",
	}
}

func queryPool(ctx context.Context, pool interface{}, relays []string, filter nostr.Filter) ([]*nostr.Event, error) {
	if events, ok, err := tryQuerySync(ctx, pool, relays, filter); ok {
		return events, err
	}
	if events, ok, err := tryQuery(ctx, pool, relays, filter); ok {
		return events, err
	}
	if events, ok, err := trySub(ctx, pool, relays, filter); ok {
		return events, err
	}
	return nil, fmt.Errorf("nostr pool does not support query")
}

func tryQuerySync(ctx context.Context, pool interface{}, relays []string, filter nostr.Filter) ([]*nostr.Event, bool, error) {
	method := reflect.ValueOf(pool).MethodByName("QuerySync")
	if !method.IsValid() {
		return nil, false, nil
	}

	results := method.Call([]reflect.Value{
		reflect.ValueOf(ctx),
		reflect.ValueOf(relays),
		reflect.ValueOf(filter),
	})

	if len(results) != 2 {
		return nil, false, fmt.Errorf("unexpected QuerySync signature")
	}

	if !results[1].IsNil() {
		if err, ok := results[1].Interface().(error); ok {
			return nil, true, err
		}
		return nil, true, fmt.Errorf("QuerySync returned non-error")
	}

	events, err := extractEvents(results[0])
	return events, true, err
}

func tryQuery(ctx context.Context, pool interface{}, relays []string, filter nostr.Filter) ([]*nostr.Event, bool, error) {
	method := reflect.ValueOf(pool).MethodByName("Query")
	if !method.IsValid() {
		return nil, false, nil
	}

	results := method.Call([]reflect.Value{
		reflect.ValueOf(ctx),
		reflect.ValueOf(relays),
		reflect.ValueOf(filter),
	})
	if len(results) != 1 {
		return nil, false, fmt.Errorf("unexpected Query signature")
	}

	ch := results[0]
	if ch.Kind() != reflect.Chan {
		return nil, false, fmt.Errorf("Query did not return channel")
	}

	events := make([]*nostr.Event, 0)
	for {
		cases := []reflect.SelectCase{
			{Dir: reflect.SelectRecv, Chan: ch},
			{Dir: reflect.SelectRecv, Chan: reflect.ValueOf(ctx.Done())},
		}
		chosen, recv, ok := reflect.Select(cases)
		if chosen == 1 {
			return events, true, nil
		}
		if !ok {
			break
		}
		evts, err := extractEvents(recv)
		if err != nil {
			return events, true, err
		}
		events = append(events, evts...)
	}

	return events, true, nil
}

func extractEvents(value reflect.Value) ([]*nostr.Event, error) {
	if !value.IsValid() {
		return nil, nil
	}

	if value.Kind() == reflect.Slice {
		events := make([]*nostr.Event, 0, value.Len())
		for i := 0; i < value.Len(); i++ {
			item := value.Index(i)
			if item.Kind() == reflect.Ptr && item.Type() == reflect.TypeOf(&nostr.Event{}) {
				events = append(events, item.Interface().(*nostr.Event))
				continue
			}
			if item.Type() == reflect.TypeOf(nostr.Event{}) {
				evt := item.Interface().(nostr.Event)
				events = append(events, &evt)
				continue
			}
		}
		return events, nil
	}

	if value.Kind() == reflect.Ptr && value.Type() == reflect.TypeOf(&nostr.Event{}) {
		return []*nostr.Event{value.Interface().(*nostr.Event)}, nil
	}
	if value.Type() == reflect.TypeOf(nostr.Event{}) {
		evt := value.Interface().(nostr.Event)
		return []*nostr.Event{&evt}, nil
	}

	return nil, fmt.Errorf("unsupported event type: %s", value.Type().String())
}

func trySub(ctx context.Context, pool interface{}, relays []string, filter nostr.Filter) ([]*nostr.Event, bool, error) {
	method := reflect.ValueOf(pool).MethodByName("Sub")
	if !method.IsValid() {
		return nil, false, nil
	}

	args := []reflect.Value{reflect.ValueOf(ctx), reflect.ValueOf(relays)}
	methodType := method.Type()
	if methodType.NumIn() >= 3 {
		lastParam := methodType.In(2)
		if methodType.IsVariadic() {
			args = append(args, reflect.ValueOf(filter))
		} else if lastParam.Kind() == reflect.Slice && lastParam.Elem() == reflect.TypeOf(nostr.Filter{}) {
			args = append(args, reflect.ValueOf([]nostr.Filter{filter}))
		} else {
			args = append(args, reflect.ValueOf(filter))
		}
	}

	results := method.Call(args)
	if len(results) != 1 {
		return nil, true, fmt.Errorf("unexpected Sub signature")
	}

	sub := results[0]
	events, err := collectFromSubscription(ctx, sub)
	unsubscribe(sub)
	return events, true, err
}

func collectFromSubscription(ctx context.Context, sub reflect.Value) ([]*nostr.Event, error) {
	eventsChan, eoseChan, err := extractSubscriptionChannels(sub)
	if err != nil {
		return nil, err
	}

	events := make([]*nostr.Event, 0)
	eventCase := reflect.SelectCase{Dir: reflect.SelectRecv, Chan: eventsChan}
	var eoseCase *reflect.SelectCase
	if eoseChan.IsValid() {
		caseCopy := reflect.SelectCase{Dir: reflect.SelectRecv, Chan: eoseChan}
		eoseCase = &caseCopy
	}

	for {
		cases := []reflect.SelectCase{
			eventCase,
			{Dir: reflect.SelectRecv, Chan: reflect.ValueOf(ctx.Done())},
		}
		if eoseCase != nil {
			cases = append(cases, *eoseCase)
		}

		chosen, recv, ok := reflect.Select(cases)
		if chosen == 1 {
			return events, nil
		}
		if eoseCase != nil && chosen == 2 {
			return events, nil
		}
		if !ok {
			return events, nil
		}

		if recv.Kind() == reflect.Ptr && recv.Type() == reflect.TypeOf(&nostr.Event{}) {
			events = append(events, recv.Interface().(*nostr.Event))
			continue
		}
		if recv.Type() == reflect.TypeOf(nostr.Event{}) {
			evt := recv.Interface().(nostr.Event)
			events = append(events, &evt)
		}
	}
}

func extractSubscriptionChannels(sub reflect.Value) (reflect.Value, reflect.Value, error) {
	if sub.Kind() == reflect.Ptr {
		sub = sub.Elem()
	}
	if !sub.IsValid() {
		return reflect.Value{}, reflect.Value{}, fmt.Errorf("invalid subscription")
	}

	eventsField := sub.FieldByName("Events")
	if !eventsField.IsValid() || eventsField.Kind() != reflect.Chan {
		return reflect.Value{}, reflect.Value{}, fmt.Errorf("subscription missing Events channel")
	}

	eoseField := sub.FieldByName("EOSE")
	if !eoseField.IsValid() {
		eoseField = sub.FieldByName("EndOfStoredEvents")
	}

	return eventsField, eoseField, nil
}

func unsubscribe(sub reflect.Value) {
	unsub := sub.MethodByName("Unsub")
	if !unsub.IsValid() {
		if sub.Kind() == reflect.Ptr {
			unsub = sub.Elem().MethodByName("Unsub")
		}
	}
	if unsub.IsValid() {
		unsub.Call(nil)
	}
}

func subscribeRelay(ctx context.Context, relay interface{}, filter nostr.Filter) (interface{}, error) {
	method := reflect.ValueOf(relay).MethodByName("Subscribe")
	if !method.IsValid() {
		return nil, fmt.Errorf("relay does not support Subscribe")
	}

	args := []reflect.Value{reflect.ValueOf(ctx)}
	methodType := method.Type()

	if methodType.NumIn() >= 2 {
		paramType := methodType.In(1)
		if value, ok := buildFiltersValue(paramType, filter); ok {
			args = append(args, value)
		} else if paramType == reflect.TypeOf(nostr.Filter{}) {
			args = append(args, reflect.ValueOf(filter))
		}
	}

	var results []reflect.Value
	results = method.Call(args)
	if len(results) != 2 {
		return nil, fmt.Errorf("unexpected Subscribe signature")
	}

	if !results[1].IsNil() {
		if err, ok := results[1].Interface().(error); ok {
			return nil, err
		}
		return nil, fmt.Errorf("subscribe returned non-error")
	}

	return results[0].Interface(), nil
}

func buildFiltersValue(paramType reflect.Type, filter nostr.Filter) (reflect.Value, bool) {
	if paramType.Kind() == reflect.Slice && paramType.Elem() == reflect.TypeOf(nostr.Filter{}) {
		slice := reflect.MakeSlice(paramType, 1, 1)
		slice.Index(0).Set(reflect.ValueOf(filter))
		return slice, true
	}
	return reflect.Value{}, false
}
