package store

import (
	"context"
	"database/sql"
	"fmt"
	"time"
)

// Pin represents a CID discovered from Nostr.
type Pin struct {
	CID       string    `json:"cid"`
	EventID   string    `json:"event_id"`
	Author    string    `json:"author"`
	Type      string    `json:"type"`
	Status    string    `json:"status"`
	SizeBytes int64     `json:"size_bytes"`
	LastError string    `json:"last_error"`
	CreatedAt time.Time `json:"created_at"`
	UpdatedAt time.Time `json:"updated_at"`
}

// InsertPins inserts pins if they don't exist.
func (s *Store) InsertPins(ctx context.Context, pins []Pin) (int, error) {
	if s == nil || s.db == nil {
		return 0, fmt.Errorf("store not initialized")
	}
	if len(pins) == 0 {
		return 0, nil
	}

	query := `INSERT OR IGNORE INTO ipfstr_pins
		(cid, event_id, author, type, status, size_bytes, last_error, in_progress)
		VALUES (?, ?, ?, ?, ?, ?, ?, 0)`

	inserted := 0
	for _, pin := range pins {
		res, err := s.db.ExecContext(ctx, query, pin.CID, pin.EventID, pin.Author, pin.Type, pin.Status, pin.SizeBytes, pin.LastError)
		if err != nil {
			return inserted, fmt.Errorf("insert pin: %w", err)
		}
		rows, _ := res.RowsAffected()
		if rows > 0 {
			inserted++
		}
	}

	return inserted, nil
}

// GetRandomPendingPin returns a random pin that is not yet pinned.
func (s *Store) GetRandomPendingPin(ctx context.Context) (Pin, error) {
	if s == nil || s.db == nil {
		return Pin{}, fmt.Errorf("store not initialized")
	}

	query := `SELECT cid, event_id, author, type, status, size_bytes, last_error, created_at, updated_at
		FROM ipfstr_pins
		WHERE status NOT IN ('pinned', 'ignored') AND in_progress = 0
		ORDER BY RANDOM()
		LIMIT 1`

	var pin Pin
	var createdAt, updatedAt string
	row := s.db.QueryRowContext(ctx, query)
	if err := row.Scan(&pin.CID, &pin.EventID, &pin.Author, &pin.Type, &pin.Status, &pin.SizeBytes, &pin.LastError, &createdAt, &updatedAt); err != nil {
		return Pin{}, err
	}
	pin.CreatedAt = parseTime(createdAt)
	pin.UpdatedAt = parseTime(updatedAt)
	return pin, nil
}

// IgnorePin sets a pin to ignored to stop retrying.
func (s *Store) IgnorePin(ctx context.Context, cid string) error {
	if s == nil || s.db == nil {
		return fmt.Errorf("store not initialized")
	}

	query := `UPDATE ipfstr_pins
		SET status = 'ignored', in_progress = 0, updated_at = CURRENT_TIMESTAMP
		WHERE cid = ?`
	_, err := s.db.ExecContext(ctx, query, cid)
	if err != nil {
		return fmt.Errorf("ignore pin: %w", err)
	}
	return nil
}

// UnignorePin resets a pin to pending for retry.
func (s *Store) UnignorePin(ctx context.Context, cid string) error {
	if s == nil || s.db == nil {
		return fmt.Errorf("store not initialized")
	}

	query := `UPDATE ipfstr_pins
		SET status = 'pending', last_error = '', in_progress = 0, updated_at = CURRENT_TIMESTAMP
		WHERE cid = ?`
	_, err := s.db.ExecContext(ctx, query, cid)
	if err != nil {
		return fmt.Errorf("unignore pin: %w", err)
	}
	return nil
}

// SetPinInProgress marks a pin as in progress or idle.
func (s *Store) SetPinInProgress(ctx context.Context, cid string, inProgress bool) error {
	if s == nil || s.db == nil {
		return fmt.Errorf("store not initialized")
	}

	value := 0
	if inProgress {
		value = 1
	}

	query := `UPDATE ipfstr_pins
		SET in_progress = ?, updated_at = CURRENT_TIMESTAMP
		WHERE cid = ?`
	_, err := s.db.ExecContext(ctx, query, value, cid)
	if err != nil {
		return fmt.Errorf("update in_progress: %w", err)
	}
	return nil
}

// ResetInProgressPins clears in_progress flags after a restart.
func (s *Store) ResetInProgressPins(ctx context.Context) (int, error) {
	if s == nil || s.db == nil {
		return 0, fmt.Errorf("store not initialized")
	}

	query := `UPDATE ipfstr_pins
		SET in_progress = 0, updated_at = CURRENT_TIMESTAMP
		WHERE in_progress = 1`
	res, err := s.db.ExecContext(ctx, query)
	if err != nil {
		return 0, fmt.Errorf("reset in_progress: %w", err)
	}
	rows, _ := res.RowsAffected()
	return int(rows), nil
}

// UpdatePinResult updates status, size, and error.
func (s *Store) UpdatePinResult(ctx context.Context, cid string, status string, sizeBytes int64, lastError string) error {
	if s == nil || s.db == nil {
		return fmt.Errorf("store not initialized")
	}

	query := `UPDATE ipfstr_pins
		SET status = ?, size_bytes = ?, last_error = ?, in_progress = 0, updated_at = CURRENT_TIMESTAMP
		WHERE cid = ?`
	_, err := s.db.ExecContext(ctx, query, status, sizeBytes, lastError, cid)
	if err != nil {
		return fmt.Errorf("update pin result: %w", err)
	}
	return nil
}

// GetPinStats returns counts and sizes by type and status.
func (s *Store) GetPinStats(ctx context.Context) ([]map[string]interface{}, error) {
	if s == nil || s.db == nil {
		return nil, fmt.Errorf("store not initialized")
	}

	query := `SELECT type, status, COUNT(*), COALESCE(SUM(size_bytes), 0)
		FROM ipfstr_pins
		GROUP BY type, status`

	rows, err := s.db.QueryContext(ctx, query)
	if err != nil {
		return nil, fmt.Errorf("query stats: %w", err)
	}
	defer rows.Close()

	var results []map[string]interface{}
	for rows.Next() {
		var typ, status string
		var count int
		var totalSize int64
		if err := rows.Scan(&typ, &status, &count, &totalSize); err != nil {
			return nil, fmt.Errorf("scan stats: %w", err)
		}
		results = append(results, map[string]interface{}{
			"type":       typ,
			"status":     status,
			"count":      count,
			"total_size": totalSize,
		})
	}
	return results, nil
}

// ListPins returns recent pins with pagination.
func (s *Store) ListPins(ctx context.Context, limit int, offset int) ([]Pin, error) {
	if s == nil || s.db == nil {
		return nil, fmt.Errorf("store not initialized")
	}

	query := `SELECT cid, event_id, author, type, status, size_bytes, last_error, created_at, updated_at
		FROM ipfstr_pins
		ORDER BY created_at DESC
		LIMIT ? OFFSET ?`

	rows, err := s.db.QueryContext(ctx, query, limit, offset)
	if err != nil {
		return nil, fmt.Errorf("list pins: %w", err)
	}
	defer rows.Close()

	var pins []Pin
	for rows.Next() {
		var pin Pin
		var createdAt, updatedAt string
		if err := rows.Scan(&pin.CID, &pin.EventID, &pin.Author, &pin.Type, &pin.Status, &pin.SizeBytes, &pin.LastError, &createdAt, &updatedAt); err != nil {
			return nil, fmt.Errorf("scan pins: %w", err)
		}
		pin.CreatedAt = parseTime(createdAt)
		pin.UpdatedAt = parseTime(updatedAt)
		pins = append(pins, pin)
	}
	return pins, nil
}

// CountPins returns total pin count.
func (s *Store) CountPins(ctx context.Context) (int, error) {
	if s == nil || s.db == nil {
		return 0, fmt.Errorf("store not initialized")
	}

	var total int
	if err := s.db.QueryRowContext(ctx, `SELECT COUNT(*) FROM ipfstr_pins`).Scan(&total); err != nil {
		return 0, fmt.Errorf("count pins: %w", err)
	}
	return total, nil
}

// SetState stores a key/value pair.
func (s *Store) SetState(ctx context.Context, key string, value string) error {
	if s == nil || s.db == nil {
		return fmt.Errorf("store not initialized")
	}

	query := `INSERT OR REPLACE INTO ipfstr_state (key, value) VALUES (?, ?)`
	_, err := s.db.ExecContext(ctx, query, key, value)
	if err != nil {
		return fmt.Errorf("set state: %w", err)
	}
	return nil
}

// GetState loads a key/value pair.
func (s *Store) GetState(ctx context.Context, key string) (string, error) {
	if s == nil || s.db == nil {
		return "", fmt.Errorf("store not initialized")
	}

	var value string
	query := `SELECT value FROM ipfstr_state WHERE key = ?`
	if err := s.db.QueryRowContext(ctx, query, key).Scan(&value); err != nil {
		if err == sql.ErrNoRows {
			return "", nil
		}
		return "", err
	}
	return value, nil
}

func parseTime(value string) time.Time {
	if value == "" {
		return time.Time{}
	}
	parsed, err := time.Parse(time.RFC3339Nano, value)
	if err == nil {
		return parsed
	}
	parsed, err = time.Parse("2006-01-02 15:04:05", value)
	if err == nil {
		return parsed
	}
	return time.Time{}
}
