package store

import (
	"context"
	"database/sql"
	"fmt"
	"time"
)

// Upload represents a stored upload.
type Upload struct {
	Sha256    string `json:"sha256"`
	CID       string `json:"cid"`
	Filename  string `json:"filename"`
	Extension string `json:"extension"`
	MimeType  string `json:"mime_type"`
	SizeBytes int64  `json:"size_bytes"`
}

// UploadRecord includes timestamps for listing.
type UploadRecord struct {
	Sha256    string    `json:"sha256"`
	CID       string    `json:"cid"`
	Filename  string    `json:"filename"`
	Extension string    `json:"extension"`
	MimeType  string    `json:"mime_type"`
	SizeBytes int64     `json:"size_bytes"`
	CreatedAt time.Time `json:"created_at"`
}

// Store wraps database access.
type Store struct {
	db *sql.DB
}

// Open initializes the SQLite store.
func Open(path string) (*Store, error) {
	db, err := sql.Open("sqlite3", path)
	if err != nil {
		return nil, fmt.Errorf("open sqlite: %w", err)
	}

	if err := db.Ping(); err != nil {
		return nil, fmt.Errorf("ping sqlite: %w", err)
	}

	if err := initSchema(db); err != nil {
		return nil, err
	}

	return &Store{db: db}, nil
}

// Close releases database resources.
func (s *Store) Close() error {
	if s == nil || s.db == nil {
		return nil
	}
	return s.db.Close()
}

// UpsertUpload stores or updates upload metadata and the blossom mapping table.
func (s *Store) UpsertUpload(ctx context.Context, upload Upload) error {
	if s == nil || s.db == nil {
		return fmt.Errorf("store not initialized")
	}

	tx, err := s.db.BeginTx(ctx, nil)
	if err != nil {
		return fmt.Errorf("begin tx: %w", err)
	}
	defer func() {
		if err != nil {
			_ = tx.Rollback()
		}
	}()

	mappingQuery := `INSERT OR REPLACE INTO ipfs_blossom_mapping (sha256, ipfs_cid, extension) VALUES (?, ?, ?)`
	if _, err = tx.ExecContext(ctx, mappingQuery, upload.Sha256, upload.CID, upload.Extension); err != nil {
		return fmt.Errorf("insert mapping: %w", err)
	}

	uploadQuery := `
		INSERT OR REPLACE INTO ipfstr_uploads
		(sha256, ipfs_cid, filename, extension, mime_type, size_bytes)
		VALUES (?, ?, ?, ?, ?, ?)`
	if _, err = tx.ExecContext(ctx, uploadQuery, upload.Sha256, upload.CID, upload.Filename, upload.Extension, upload.MimeType, upload.SizeBytes); err != nil {
		return fmt.Errorf("insert upload: %w", err)
	}

	if err = tx.Commit(); err != nil {
		return fmt.Errorf("commit tx: %w", err)
	}

	return nil
}

// UpsertMapping stores sha256 -> cid mapping.
func (s *Store) UpsertMapping(ctx context.Context, sha256 string, cid string, extension string) error {
	if s == nil || s.db == nil {
		return fmt.Errorf("store not initialized")
	}

	query := `INSERT OR REPLACE INTO ipfs_blossom_mapping (sha256, ipfs_cid, extension) VALUES (?, ?, ?)`
	if _, err := s.db.ExecContext(ctx, query, sha256, cid, extension); err != nil {
		return fmt.Errorf("insert mapping: %w", err)
	}
	return nil
}

// GetMapping returns the cid and extension for a sha256.
func (s *Store) GetMapping(ctx context.Context, sha256 string) (string, string, error) {
	if s == nil || s.db == nil {
		return "", "", fmt.Errorf("store not initialized")
	}

	var cid, ext string
	query := `SELECT ipfs_cid, extension FROM ipfs_blossom_mapping WHERE sha256 = ?`
	if err := s.db.QueryRowContext(ctx, query, sha256).Scan(&cid, &ext); err != nil {
		return "", "", err
	}
	return cid, ext, nil
}

// ListUploads returns recent uploads with pagination.
func (s *Store) ListUploads(ctx context.Context, limit int, offset int) ([]UploadRecord, error) {
	if s == nil || s.db == nil {
		return nil, fmt.Errorf("store not initialized")
	}

	query := `SELECT sha256, ipfs_cid, filename, extension, mime_type, size_bytes, created_at
		FROM ipfstr_uploads
		ORDER BY created_at DESC
		LIMIT ? OFFSET ?`

	rows, err := s.db.QueryContext(ctx, query, limit, offset)
	if err != nil {
		return nil, fmt.Errorf("list uploads: %w", err)
	}
	defer rows.Close()

	var uploads []UploadRecord
	for rows.Next() {
		var upload UploadRecord
		var createdAt string
		if err := rows.Scan(&upload.Sha256, &upload.CID, &upload.Filename, &upload.Extension, &upload.MimeType, &upload.SizeBytes, &createdAt); err != nil {
			return nil, fmt.Errorf("scan uploads: %w", err)
		}
		upload.CreatedAt = parseTime(createdAt)
		uploads = append(uploads, upload)
	}
	return uploads, nil
}

// CountUploads returns total uploads.
func (s *Store) CountUploads(ctx context.Context) (int, error) {
	if s == nil || s.db == nil {
		return 0, fmt.Errorf("store not initialized")
	}

	var total int
	if err := s.db.QueryRowContext(ctx, `SELECT COUNT(*) FROM ipfstr_uploads`).Scan(&total); err != nil {
		return 0, fmt.Errorf("count uploads: %w", err)
	}
	return total, nil
}

func initSchema(db *sql.DB) error {
	queries := []string{
		`CREATE TABLE IF NOT EXISTS ipfs_blossom_mapping (
			sha256 TEXT PRIMARY KEY,
			ipfs_cid TEXT NOT NULL,
			extension TEXT,
			created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
		);`,
		`CREATE TABLE IF NOT EXISTS ipfstr_uploads (
			sha256 TEXT PRIMARY KEY,
			ipfs_cid TEXT NOT NULL,
			filename TEXT,
			extension TEXT,
			mime_type TEXT,
			size_bytes INTEGER,
			created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
		);`,
		`CREATE TABLE IF NOT EXISTS ipfstr_pins (
			cid TEXT PRIMARY KEY,
			event_id TEXT,
			author TEXT,
			type TEXT,
			status TEXT,
			size_bytes INTEGER,
			last_error TEXT,
			in_progress INTEGER DEFAULT 0,
			created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
			updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
		);`,
		`CREATE INDEX IF NOT EXISTS idx_ipfstr_pins_status ON ipfstr_pins(status);`,
		`CREATE INDEX IF NOT EXISTS idx_ipfstr_pins_in_progress ON ipfstr_pins(in_progress);`,
		`CREATE TABLE IF NOT EXISTS ipfstr_state (
			key TEXT PRIMARY KEY,
			value TEXT
		);`,
	}

	for _, query := range queries {
		if _, err := db.Exec(query); err != nil {
			return fmt.Errorf("init schema: %w", err)
		}
	}

	return nil
}
